<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\Setting; // For fetching setting groups
use App\Models\ModulePermission;
use App\Models\RoleSettingGroupPermission; // Correct model for setting group permissions
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Nwidart\Modules\Facades\Module;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB; // Added for database transactions
use Illuminate\Support\Facades\Log; // Added for logging errors

class RoleController extends Controller
{
    public function __construct()
    {
        // Protect all methods in this controller.
        // Ensure you have a 'manage roles' permission defined and assigned appropriately,
        // or restrict to 'super_admin' role if that's your policy.
        $this->middleware('can:manage roles');
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Eager load Spatie permissions and also fetch module permissions separately
        $roles = Role::with(['permissions', 'users'])->orderBy('name')->paginate(15); // Eager load users for count

        // For each role, fetch its module permissions and setting group permissions
        // This is not ideal for N+1 if you have many roles on a page,
        // but for a paginated list of 15, it's acceptable.
        // A more optimized way would be to fetch all module permissions and map them.
        $roles->each(function ($role) {
            $role->module_access_permissions = ModulePermission::where('role_name', $role->name)->pluck('module_name')->all();
            $settingGroupPerms = RoleSettingGroupPermission::where('role_id', $role->id)
                ->where(function ($query) {
                    $query->where('can_view', true)->orWhere('can_edit', true);
                })
                ->pluck('setting_group_name');
            $role->setting_group_access_summary = $settingGroupPerms->isNotEmpty() ? $settingGroupPerms->all() : [];
            // Add users_count directly to the role object
            $role->users_count = $role->users->count(); // Access the loaded users relationship
        });
        return view('admin.roles.index', compact('roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $permissionsFromDb = Permission::orderBy('name')->get();
        $enabledModules = Module::allEnabled();
        $moduleLookup = [];
        foreach ($enabledModules as $module) {
            $moduleName = $module->getName();
            $moduleDisplayName = $module->get('display_name', $moduleName);
            // Store various forms for matching (lowercase)
            $moduleLookup[Str::lower($moduleName)] = $moduleDisplayName;
            $alias = $module->get('alias'); // Get alias from module.json
            if ($alias) {
                $moduleLookup[Str::lower($alias)] = $moduleDisplayName;
            }
        }

        $permissions = []; // This will be the grouped structure
        foreach ($permissionsFromDb as $permission) {
            $parts = preg_split('/[\s_.]+/', $permission->name, 2); // Split by space, underscore, or dot for the first part
            $potentialModuleKey = Str::lower($parts[0]);
            $groupName = 'General Permissions'; // Default group

            if (isset($moduleLookup[$potentialModuleKey])) {
                $groupName = $moduleLookup[$potentialModuleKey];
            }
            $permissions[$groupName][] = $permission;
        }
        ksort($permissions); // Sort groups alphabetically
        
        $allModules = Module::allEnabled(); // Fetch modules for create view as well

        return view('admin.roles.create', compact('permissions', 'allModules'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:100',
                Rule::unique('roles', 'name')->where(function ($query) {
                    return $query->where('guard_name', 'web'); // Ensure uniqueness for the 'web' guard
                }),
            ],
            'show_on_registration' => 'nullable|boolean',
            'permissions' => 'nullable|array',
            'permissions.*' => 'integer|exists:permissions,id',
            'module_access' => 'nullable|array',
            'module_access.*' => 'string', // Module names, consider custom 'exists_module' rule
            'setting_group_permissions' => 'nullable|array',
            'setting_group_permissions.*.can_view' => 'nullable|boolean',
            'setting_group_permissions.*.can_edit' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            $role = Role::create([
                'name' => $validated['name'],
                'guard_name' => 'web',
                'show_on_registration' => $request->input('show_on_registration', 0) ? 1 : 0,
            ]);

            // Sync Spatie Permissions
            if (!empty($validated['permissions'])) {
                $role->syncPermissions($validated['permissions']);
            } else {
                $role->syncPermissions([]); // Clear all Spatie permissions if none are sent
            }

            // Handle Module Access Permissions (uses role_name)
            // For a new role, no need to delete existing, but good practice if this logic is reused.
            // ModulePermission::where('role_name', $role->name)->delete(); 
            if ($request->has('module_access')) {
                $uniqueModuleNames = array_unique($request->input('module_access', []));
                foreach ($uniqueModuleNames as $moduleName) {
                    if (Module::find($moduleName)) { // Ensure module exists
                        ModulePermission::create([
                            'role_name' => $role->name,
                            'module_name' => $moduleName,
                        ]);
                    }
                }
            }

            // Handle Setting Group Access Permissions (uses role_id)
            // RoleSettingGroupPermission::where('role_id', $role->id)->delete(); 
            if ($request->has('setting_group_permissions')) {
                foreach ($request->input('setting_group_permissions') as $groupName => $groupPermissions) {
                    RoleSettingGroupPermission::create([
                        'role_id' => $role->id,
                        'setting_group_name' => $groupName,
                        'can_view' => !empty($groupPermissions['can_view']),
                        'can_edit' => !empty($groupPermissions['can_edit']),
                    ]);
                }
            }

            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating role: ' . $e->getMessage(), [
                'trace' => Str::limit($e->getTraceAsString(), 1000)
            ]);
            return redirect()->back()->withInput()->with('error', 'Failed to create role. Please try again.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Role $role)
    {
        // Typically, for roles, edit is more common than show.
        // You can implement this if needed, or redirect to edit.
        return redirect()->route('admin.roles.edit', $role);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Role $role)
    {
        $permissionsFromDb = Permission::orderBy('name')->get();
        $enabledModules = Module::allEnabled();
        $moduleLookup = [];
        foreach ($enabledModules as $module) {
            $moduleName = $module->getName();
            $moduleDisplayName = $module->get('display_name', $moduleName);
            $moduleLookup[Str::lower($moduleName)] = $moduleDisplayName;
            $alias = $module->get('alias'); // Get alias from module.json
            if ($alias) {
                $moduleLookup[Str::lower($alias)] = $moduleDisplayName;
            }
        }

        $permissions = [];
        foreach ($permissionsFromDb as $permission) {
            $parts = preg_split('/[\s_.]+/', $permission->name, 2);
            $potentialModuleKey = Str::lower($parts[0]);
            $groupName = 'General Permissions';

            if (isset($moduleLookup[$potentialModuleKey])) {
                $groupName = $moduleLookup[$potentialModuleKey];
            }
            $permissions[$groupName][] = $permission;
        }
        ksort($permissions);
        $rolePermissions = $role->permissions->pluck('id')->toArray();
        
        // Data for managing module and setting group access for this role
        $allModules = Module::allEnabled();
        $roleModulePermissions = ModulePermission::where('role_name', $role->name)->pluck('module_name')->toArray();

        $allSettingGroups = Setting::distinct()->orderBy('group')->pluck('group');
        $roleSettingGroupPermissions = RoleSettingGroupPermission::where('role_id', $role->id)->get()->keyBy('setting_group_name');

        return view('admin.roles.edit', compact(
            'role', 
            'permissions', 
            'rolePermissions',
            'allModules',
            'roleModulePermissions',
            'allSettingGroups',
            'roleSettingGroupPermissions'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Role $role)
    {
        // Prevent editing of core roles like 'super_admin' or 'admin' if needed
        if (in_array($role->name, ['super_admin', 'admin']) && $request->input('name') !== $role->name) {
             return redirect()->back()->with('error', 'Cannot rename core system roles.');
        }

        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:100',
                Rule::unique('roles', 'name')->where(function ($query) {
                    return $query->where('guard_name', 'web');
                })->ignore($role->id),
            ],
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
            'show_on_registration' => 'nullable|boolean',
            'module_access' => 'nullable|array',
            'module_access.*' => 'string', // Module names
            'setting_group_permissions' => 'nullable|array',
            'setting_group_permissions.*.can_view' => 'nullable|boolean',
            'setting_group_permissions.*.can_edit' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            $oldRoleName = $role->name;
            $newRoleName = $validated['name'];

            $role->name = $newRoleName;
            $role->show_on_registration = $request->input('show_on_registration', 0) ? 1 : 0;
            $role->save();

            // Sync Spatie Permissions
            $permissionIds = $validated['permissions'] ?? [];
            if (!empty($permissionIds)) {
                $permissionsToSync = Permission::whereIn('id', $permissionIds)->get();
                $role->syncPermissions($permissionsToSync);
            } else {
                $role->syncPermissions([]); // Remove all permissions if none are provided
            }

            // Sync Module Permissions (uses role_name)
            // If role name changed, update existing entries that used the old name.
            if ($oldRoleName !== $newRoleName) {
                ModulePermission::where('role_name', $oldRoleName)->update(['role_name' => $newRoleName]);
            }
            // Clear and re-add based on current selection for the (potentially new) role name
            ModulePermission::where('role_name', $newRoleName)->delete();
            if ($request->has('module_access')) {
                $uniqueModuleNames = array_unique($request->input('module_access', []));
                foreach ($uniqueModuleNames as $moduleName) {
                    if (Module::find($moduleName)) { // Ensure module exists
                        ModulePermission::create([
                            'role_name' => $newRoleName,
                            'module_name' => $moduleName,
                        ]);
                    }
                }
            }

            // Sync Setting Group Permissions
            // Uses role_id, so role name change doesn't affect the foreign key.
            RoleSettingGroupPermission::where('role_id', $role->id)->delete(); // Clear existing for this role

            if ($request->has('setting_group_permissions')) {
                foreach ($request->input('setting_group_permissions') as $groupName => $groupPermissions) {
                    RoleSettingGroupPermission::create([
                        'role_id' => $role->id,
                        'setting_group_name' => $groupName,
                        'can_view' => !empty($groupPermissions['can_view']),
                        'can_edit' => !empty($groupPermissions['can_edit']),
                    ]);
                }
            }
            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating role: ' . $e->getMessage(), [
                'trace' => Str::limit($e->getTraceAsString(), 1000)
            ]);
            return redirect()->back()->withInput()->with('error', 'Failed to update role. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role)
    {
        // Prevent deletion of core roles
        if (in_array($role->name, ['super_admin', 'admin', 'user'])) {
            return redirect()->route('admin.roles.index')->with('error', 'Cannot delete core system roles.');
        }

        // Consider what happens to users with this role.
        // Spatie doesn't automatically reassign users. You might need custom logic here.
        if ($role->users()->count() > 0) {
            return redirect()->route('admin.roles.index')->with('error', 'Cannot delete role: users are currently assigned to this role.');
        }

        DB::beginTransaction();
        try {
            // Delete related custom permissions first
            ModulePermission::where('role_name', $role->name)->delete();
            RoleSettingGroupPermission::where('role_id', $role->id)->delete();

            $role->delete(); // Spatie will handle its related permissions
            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting role: ' . $e->getMessage());
            return redirect()->route('admin.roles.index')->with('error', 'Failed to delete role.');
        }
    }
}