<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NewsletterSubscription;
use Illuminate\Http\Request;

class NewsletterSubscriptionController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->isSuperAdmin() && !auth()->user()->can('manage newsletter')) { // Example permission
                abort(403, 'Unauthorized action.');
            }
            return $next($request);
        });
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = NewsletterSubscription::orderBy('created_at', 'desc');
        if ($request->filled('search')) {
            $query->where('email', 'like', '%' . $request->search . '%');
        }
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        $subscriptions = $query->paginate(20);
        $statuses = ['subscribed', 'unsubscribed', 'pending'];
        return view('admin.newsletter_subscriptions.index', compact('subscriptions', 'statuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.newsletter_subscriptions.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|max:255|unique:newsletter_subscriptions,email',
            'status' => 'required|in:subscribed,unsubscribed,pending',
        ]);

        NewsletterSubscription::create($validated + ['subscribed_at' => $validated['status'] === 'subscribed' ? now() : null]);

        return redirect()->route('admin.newsletter-subscriptions.index')->with('success', 'Subscription created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(NewsletterSubscription $newsletterSubscription)
    {
        return view('admin.newsletter_subscriptions.edit', compact('newsletterSubscription'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, NewsletterSubscription $newsletterSubscription)
    {
        $validated = $request->validate([
            'email' => 'required|email|max:255|unique:newsletter_subscriptions,email,' . $newsletterSubscription->id,
            'status' => 'required|in:subscribed,unsubscribed,pending',
        ]);

        $data = $validated;
        if ($validated['status'] === 'subscribed' && $newsletterSubscription->status !== 'subscribed') {
            $data['subscribed_at'] = now();
            $data['unsubscribed_at'] = null;
        } elseif ($validated['status'] === 'unsubscribed' && $newsletterSubscription->status !== 'unsubscribed') {
            $data['unsubscribed_at'] = now();
        }
        $newsletterSubscription->update($data);
        return redirect()->route('admin.newsletter-subscriptions.index')->with('success', 'Subscription updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(NewsletterSubscription $newsletterSubscription)
    {
        $newsletterSubscription->delete();
        return redirect()->route('admin.newsletter-subscriptions.index')->with('success', 'Subscription deleted successfully.');
    }
}
