<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Newsletter;
use App\Models\NewsletterSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Mail\NewsletterMail; // We will create this mailable later
use Illuminate\Support\Facades\Mail;
use App\Jobs\SendNewsletterJob; // We will create this job later
use Illuminate\Support\Facades\Log;

class NewsletterController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            // Ensure user has permission to manage newsletters
            if (!Auth::user()->isSuperAdmin() && !Auth::user()->can('manage newsletters')) { // Example permission
                abort(403, 'Unauthorized action.');
            }
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $query = Newsletter::orderBy('created_at', 'desc');
        if ($request->filled('search')) {
            $query->where('subject', 'like', '%' . $request->search . '%');
        }
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        $newsletters = $query->paginate(15);
        $statuses = ['draft', 'sending', 'sent', 'failed'];
        return view('admin.newsletters.index', compact('newsletters', 'statuses'));
    }

    public function create()
    {
        return view('admin.newsletters.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        Newsletter::create([
            'subject' => $validated['subject'],
            'content' => $validated['content'],
            'status' => 'draft', // Default to draft
        ]);

        return redirect()->route('admin.newsletters.index')->with('success', 'Newsletter created as draft.');
    }

    public function show(Newsletter $newsletter)
    {
        // For previewing the newsletter content
        return view('admin.newsletters.show', compact('newsletter'));
    }

    public function edit(Newsletter $newsletter)
    {
        if ($newsletter->status === 'sent' || $newsletter->status === 'sending') {
            return redirect()->route('admin.newsletters.index')->with('error', 'Cannot edit a newsletter that is sending or already sent.');
        }
        return view('admin.newsletters.edit', compact('newsletter'));
    }

    public function update(Request $request, Newsletter $newsletter)
    {
        if ($newsletter->status === 'sent' || $newsletter->status === 'sending') {
            return redirect()->route('admin.newsletters.index')->with('error', 'Cannot update a newsletter that is sending or already sent.');
        }

        $validated = $request->validate([
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        $newsletter->update($validated);

        return redirect()->route('admin.newsletters.index')->with('success', 'Newsletter updated successfully.');
    }

    public function destroy(Newsletter $newsletter)
    {
        if ($newsletter->status === 'sending') {
            return redirect()->route('admin.newsletters.index')->with('error', 'Cannot delete a newsletter that is currently sending.');
        }
        $newsletter->delete();
        return redirect()->route('admin.newsletters.index')->with('success', 'Newsletter deleted successfully.');
    }

    public function sendTest(Request $request, Newsletter $newsletter)
    {
        $request->validate(['test_email' => 'required|email']);
        try {
            Mail::to($request->test_email)->send(new NewsletterMail($newsletter->subject, $newsletter->content));
            return back()->with('success', 'Test email sent to ' . $request->test_email);
        } catch (\Exception $e) {
            Log::error("Failed to send test newsletter: " . $e->getMessage());
            return back()->with('error', 'Failed to send test email. Check mail configuration and logs.');
        }
    }

    public function dispatchSend(Newsletter $newsletter)
    {
        if ($newsletter->status !== 'draft') {
            return redirect()->route('admin.newsletters.index')->with('error', 'Newsletter is not in draft status or already sent/sending.');
        }

        $subscribersCount = NewsletterSubscription::where('status', 'subscribed')->count();
        if ($subscribersCount === 0) {
            return redirect()->route('admin.newsletters.index')->with('info', 'No active subscribers to send the newsletter to.');
        }

        $newsletter->update(['status' => 'sending', 'recipients_count' => $subscribersCount, 'sent_by_user_id' => Auth::id()]);
        SendNewsletterJob::dispatch($newsletter); // Dispatch the job

        return redirect()->route('admin.newsletters.index')->with('success', "Newsletter '{$newsletter->subject}' is now being sent to {$subscribersCount} subscribers.");
    }
}
