<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Nwidart\Modules\Facades\Module; // Import the Module facade
use Illuminate\Http\Request;

class ModuleController extends Controller
{
    /**
     * Display the modules management page.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $modules = Module::all(); // Get all registered modules
        return view('admin.modules.index', compact('modules'));
    }

    /**
     * Show the form for installing a new module.
     *
     * @return \Illuminate\View\View
     */
    public function showInstallForm()
    {
        return view('admin.modules.install');
    }

    /**
     * Handle the upload of a new module ZIP file.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function uploadModule(Request $request)
    {
        $request->validate([
            'module_zip' => 'required|file|mimes:zip|max:20480', // Max 20MB, ensure it's a zip
        ]);

        $uploadedFile = $request->file('module_zip');
        $tempZipPath = null;
        $tempExtractBasePath = storage_path('app/temp_modules_extract'); // Base path for extractions
        $specificTempExtractPath = null; // Path for this specific extraction

        try {
            if (!$uploadedFile->isValid()) {
                return redirect()->back()->with('error', 'Invalid file uploaded.');
            }

            $fileName = time() . '_' . $uploadedFile->getClientOriginalName();
            // Store the uploaded ZIP file temporarily in storage/app/temp_modules
            $storedZipRelativePath = $uploadedFile->storeAs('temp_modules', $fileName, 'local');
            $tempZipPath = storage_path('app/' . $storedZipRelativePath);

            // Basic extraction and installation (can be made more robust)
            $zip = new \ZipArchive;
            if ($zip->open($tempZipPath) !== TRUE) {
                return redirect()->back()->with('error', 'Could not open the module ZIP file. It might be corrupted or not a valid ZIP.');
            }

                // Extract to a temporary directory first to inspect
            $specificTempExtractPath = $tempExtractBasePath . '/' . time() . '_' . Str::random(8);
            File::ensureDirectoryExists($specificTempExtractPath);
            $zip->extractTo($specificTempExtractPath);
            $zip->close(); // Close the archive as soon as possible

                // Assume the first directory in the extracted path is the module itself
            $extractedFolders = File::directories($specificTempExtractPath);

            if (empty($extractedFolders)) {
                return redirect()->back()->with('error', 'The ZIP file does not seem to contain a module directory.');
            }
            if (count($extractedFolders) > 1) {
                return redirect()->back()->with('error', 'The ZIP file should contain only a single root module directory.');
            }

            $moduleSourcePath = $extractedFolders[0];
            $moduleName = basename($moduleSourcePath);

                    // Validate: Check for module.json
                if (!File::exists($moduleSourcePath . '/module.json')) {
                    return redirect()->back()->with('error', "Invalid module structure: module.json not found in the root of '{$moduleName}' directory.");
                    }

                    // Move the extracted module to the Modules directory
                $destinationPath = base_path('Modules/' . $moduleName);
                if (File::exists($destinationPath)) {
                    return redirect()->back()->with('error', "Module '{$moduleName}' already exists. Please uninstall the existing one first or use a different name.");
                    }

                // Attempt to move the directory
                if (!File::moveDirectory($moduleSourcePath, $destinationPath)) {
                    return redirect()->back()->with('error', "Could not move module '{$moduleName}' to the Modules directory. Check file permissions.");
                }

                    // Scan for new modules
                    Artisan::call('module:scan');
                    Artisan::call('optimize:clear'); // Clear caches after installing a new module

                    return redirect()->route('admin.modules.index')->with('success', "Module '{$moduleName}' uploaded and installed successfully. You may need to enable it.");

        } catch (\Exception $e) {
            // Log the exception for debugging
            Log::error("Module upload failed: " . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return redirect()->back()->with('error', 'An unexpected error occurred during module upload: ' . $e->getMessage());
        } finally {
            // Clean up temporary extraction directory
            if ($specificTempExtractPath && File::isDirectory($specificTempExtractPath)) {
                File::deleteDirectory($specificTempExtractPath);
            }
            // Clean up temporary zip file
            if ($tempZipPath && File::exists($tempZipPath)) {
                File::delete($tempZipPath);
            }
        }
    }

    /**
     * Enable the specified module.
     *
     * @param  string $moduleName
     * @return \Illuminate\Http\RedirectResponse
     */
    public function enableModule(string $moduleName)
    {
        $module = Module::find($moduleName);
        if (!$module) {
            return redirect()->route('admin.modules.index')->with('error', "Module '{$moduleName}' not found.");
        }

        $module->enable();
        Artisan::call('optimize:clear'); // Clear caches after enabling
        return redirect()->route('admin.modules.index')->with('success', "Module '{$module->getName()}' enabled successfully.");
    }

    /**
     * Disable the specified module.
     *
     * @param  string $moduleName
     * @return \Illuminate\Http\RedirectResponse
     */
    public function disableModule(string $moduleName)
    {
        $module = Module::find($moduleName);
        if (!$module) {
            return redirect()->route('admin.modules.index')->with('error', "Module '{$moduleName}' not found.");
        }

        $module->disable();
        Artisan::call('optimize:clear'); // Clear caches after disabling
        return redirect()->route('admin.modules.index')->with('success', "Module '{$module->getName()}' disabled successfully.");
    }

    /**
     * Uninstall (delete) the specified module.
     *
     * @param  string $moduleName
     * @return \Illuminate\Http\RedirectResponse
     */
    public function uninstallModule(string $moduleName)
    {
        $module = Module::find($moduleName);
        if (!$module) {
            return redirect()->route('admin.modules.index')->with('error', "Module '{$moduleName}' not found.");
        }

        // Add any pre-deletion logic here (e.g., running module-specific uninstall scripts, database cleanup)
        // This is a critical part for robust uninstallation.

        $module->delete(); // This deletes the module's files
        Artisan::call('module:scan'); // Rescan modules
        Artisan::call('optimize:clear'); // Clear caches

        return redirect()->route('admin.modules.index')->with('success', "Module '{$moduleName}' uninstalled successfully.");
    }
}