<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FeatureCreditCost;
use Illuminate\Http\Request;
use Nwidart\Modules\Facades\Module;
use Illuminate\Support\Str;

class FeatureCreditCostController extends Controller
{
    public function __construct()
    {
        // Ensure only super admins can manage this
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->isSuperAdmin()) {
                abort(403, 'Unauthorized action.');
            }
            return $next($request);
        });
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = FeatureCreditCost::query();
        if ($request->filled('search')) {
            $query->where('feature_key', 'like', '%' . $request->search . '%');
        }
        $featureCosts = $query->orderBy('feature_key')->paginate(20);
        return view('admin.feature_credit_costs.index', compact('featureCosts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $availableFeatureKeys = $this->getAvailableFeatureKeys();
        $existingKeys = FeatureCreditCost::pluck('feature_key')->toArray();
        $creatableFeatureKeys = array_diff($availableFeatureKeys, $existingKeys);

        return view('admin.feature_credit_costs.create', compact('creatableFeatureKeys'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'feature_key' => 'required|string|max:255|unique:feature_credit_costs,feature_key',
            'credit_cost' => 'required|integer|min:0',
        ]);

        FeatureCreditCost::create($request->all());

        return redirect()->route('admin.feature-credit-costs.index')
                         ->with('success', 'Feature credit cost created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FeatureCreditCost $featureCreditCost)
    {
        // No need to pass available keys for edit, as feature_key is not editable.
        return view('admin.feature_credit_costs.edit', compact('featureCreditCost'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FeatureCreditCost $featureCreditCost)
    {
        $request->validate([
            // feature_key is not updatable to maintain integrity, only cost.
            'credit_cost' => 'required|integer|min:0',
        ]);

        $featureCreditCost->update(['credit_cost' => $request->credit_cost]);

        return redirect()->route('admin.feature-credit-costs.index')
                         ->with('success', 'Feature credit cost updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FeatureCreditCost $featureCreditCost)
    {
        $featureCreditCost->delete();

        return redirect()->route('admin.feature-credit-costs.index')
                         ->with('success', 'Feature credit cost deleted successfully.');
    }

    /**
     * Helper function to get all defined feature keys from modules.
     */
    private function getAvailableFeatureKeys(): array
    {
        $keys = [];
        if (class_exists(Module::class)) {
            foreach (Module::allEnabled() as $module) {
                $moduleFeatures = $module->get('features');
                if (is_array($moduleFeatures)) {
                    foreach ($moduleFeatures as $feature) {
                        if (isset($feature['key'])) {
                            $keys[$feature['key']] = $feature['name'] . ' (' . $module->getName() . ')';
                        } else {
                            // Fallback if key is not explicitly defined in module.json feature entry
                            // This might happen if features are defined differently.
                            // For robustness, ensure 'key' is always present in your module.json feature definitions.
                            $generatedKey = Str::slug($module->getName() . ' ' . ($feature['name'] ?? 'unknown_feature_' . Str::random(4)));
                            $keys[$generatedKey] = ($feature['name'] ?? 'Unknown Feature') . ' (' . $module->getName() . ') - Generated Key';
                        }
                    }
                }
            }
        }
        // You might also want to add "global" feature keys not tied to modules
        // For example: $keys['global_premium_support'] = 'Global Premium Support Access';

        ksort($keys);
        return $keys;
    }
}