<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Nwidart\Modules\Facades\Module;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log; // For logging potential issues
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $stats = [
            'total_users' => User::count(),
            'admin_users' => User::where('is_admin', true)->count(),
            'active_modules' => count(Module::allEnabled()),
            'total_modules' => count(Module::all()),
            // Add more stats here. Modules can provide stats via a service/event.
        ];

        // Example: Last 5 registered users for recent activity
        $recentActivities = User::latest()->take(5)->get()->map(function ($user) {
            return (object) [
                'description' => "New user registered: {$user->name}",
                'created_at' => $user->created_at,
                // 'icon' => 'user-plus' // You can add icon hints for the view
            ];
        });

        // Data for User Registrations Chart (Last 7 Days)
        $userRegistrations = User::select(DB::raw('DATE(created_at) as date'), DB::raw('count(*) as count'))
            ->where('created_at', '>=', now()->subDays(6)) // Last 7 days including today
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        $userRegistrationChartData = [
            'labels' => $userRegistrations->pluck('date')->map(fn($date) => \Carbon\Carbon::parse($date)->format('M d'))->toArray(),
            'data' => $userRegistrations->pluck('count')->toArray(),
        ];

        // Initialize GA stats
        $gaStats = null;
        $gaTopPages = null;
        $landingPageGaEnabled = false;

        // Check if LandingPage module is enabled and try to get GA stats
        if (Module::has('LandingPage') && Module::isEnabled('LandingPage')) {
            // Check if the 'setting' helper function and GA settings exist
            if (function_exists('setting') &&
                setting('google_analytics_enable_tracking', false) &&
                setting('google_analytics_property_id', false) &&
                setting('google_analytics_credentials_path', false)
            ) {
                $landingPageGaEnabled = true;
                try {
                    if (app()->bound(\Modules\LandingPage\Services\GoogleAnalyticsService::class)) {
                        $gaService = app(\Modules\LandingPage\Services\GoogleAnalyticsService::class);
                        $gaStats = $gaService->getBasicStats('7daysAgo', 'today');
                        $gaTopPages = $gaService->getTopPages('7daysAgo', 'today', 5);
                    } else {
                        Log::warning('LandingPage GoogleAnalyticsService is not bound, cannot fetch GA stats.');
                        $landingPageGaEnabled = false; // Set to false if service is not available
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to retrieve Google Analytics data: ' . $e->getMessage());
                    // $gaStats and $gaTopPages will remain null, dashboard won't break.
                    // You might want to set $landingPageGaEnabled = false here too, or pass an error message to the view.
                }
            }
        }
        return view('admin.dashboard', compact('stats', 'recentActivities', 'userRegistrationChartData', 'gaStats', 'gaTopPages', 'landingPageGaEnabled'));
    }

    /**
     * Display a dashboard listing all payment gateway modules.
     *
     * @return \Illuminate\View\View
     */
    public function paymentGatewaysDashboard()
    {
        $paymentGatewayModules = collect();

        if (class_exists(Module::class)) {
            $paymentGatewayModules = collect(Module::allEnabled())
                ->filter(function ($module) {
                    return $module->get('category') === 'payment_gateway';
                })
                ->sortBy('name');
        }

        return view('admin.payment_gateways_dashboard', compact('paymentGatewayModules'));
    }
}
