<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WalletWithdrawalRequest;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Services\WalletService; // To process withdrawals

class AdminWalletWithdrawalRequestController extends Controller
{
    protected WalletService $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
        // Ensure only super admins can manage this
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->isSuperAdmin()) {
                abort(403, 'Unauthorized action. Super admin access required.');
            }
            return $next($request);
        });
    }

    /**
     * Display a listing of the wallet withdrawal requests.
     */
    public function index(Request $request)
    {
        $query = WalletWithdrawalRequest::with('user')->latest();

        if ($request->filled('search_user')) {
            $searchTerm = $request->search_user;
            $query->whereHas('user', function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%");
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $withdrawalRequests = $query->paginate(20)->withQueryString();
        $statuses = WalletWithdrawalRequest::getStatuses(); // Assuming a static method in the model

        return view('admin.wallet_withdrawal_requests.index', compact('withdrawalRequests', 'statuses'));
    }

    /**
     * Display the specified wallet withdrawal request.
     */
    public function show(WalletWithdrawalRequest $withdrawalRequest)
    {
        $withdrawalRequest->load('user');
        $statuses = WalletWithdrawalRequest::getStatuses();
        return view('admin.wallet_withdrawal_requests.show', compact('withdrawalRequest', 'statuses'));
    }

    /**
     * Update the status of the specified wallet withdrawal request.
     */
    public function updateStatus(Request $request, WalletWithdrawalRequest $withdrawalRequest)
    {
        $request->validate([
            'status' => ['required', 'string', \Illuminate\Validation\Rule::in(array_keys(WalletWithdrawalRequest::getStatuses()))],
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        $oldStatus = $withdrawalRequest->status;
        $newStatus = $request->input('status');

        // Prevent re-processing a completed or rejected request unless explicitly allowed
        if (in_array($oldStatus, [WalletWithdrawalRequest::STATUS_COMPLETED, WalletWithdrawalRequest::STATUS_REJECTED]) && $oldStatus === $newStatus) {
            // Allow updating notes even if status is same
            if ($request->filled('admin_notes')) {
                $withdrawalRequest->admin_notes = $request->admin_notes;
                $withdrawalRequest->save();
                return redirect()->route('admin.wallet-withdrawal-requests.show', $withdrawalRequest)->with('success', 'Admin notes updated.');
            }
            return redirect()->route('admin.wallet-withdrawal-requests.show', $withdrawalRequest)->with('info', 'Request is already ' . strtolower(WalletWithdrawalRequest::getStatuses()[$oldStatus]) . '.');
        }

        $withdrawalRequest->status = $newStatus;
        if ($request->filled('admin_notes')) {
            $withdrawalRequest->admin_notes = $request->admin_notes;
        }

        // If marking as 'completed', process the withdrawal
        if ($newStatus === WalletWithdrawalRequest::STATUS_COMPLETED && $oldStatus !== WalletWithdrawalRequest::STATUS_COMPLETED) {
            try {
                // This assumes your WalletService handles the actual deduction and records a transaction
                $this->walletService->processWithdrawal($withdrawalRequest);
                $withdrawalRequest->processed_at = now();
                // User notification for completion
                // $withdrawalRequest->user->notify(new WithdrawalRequestStatusUpdatedNotification($withdrawalRequest));
            } catch (\Exception $e) {
                // Revert status if processing fails
                $withdrawalRequest->status = $oldStatus; // Or a specific 'failed' status
                $withdrawalRequest->admin_notes = ($withdrawalRequest->admin_notes ? $withdrawalRequest->admin_notes . "\n" : '') . 'Processing failed: ' . $e->getMessage();
                $withdrawalRequest->save();
                return redirect()->route('admin.wallet-withdrawal-requests.show', $withdrawalRequest)->with('error', 'Failed to process withdrawal: ' . $e->getMessage());
            }
        } elseif ($newStatus === WalletWithdrawalRequest::STATUS_REJECTED && $oldStatus !== WalletWithdrawalRequest::STATUS_REJECTED) {
            // User notification for rejection
            // $withdrawalRequest->user->notify(new WithdrawalRequestStatusUpdatedNotification($withdrawalRequest));
        }

        $withdrawalRequest->save();

        // General user notification for status change (if not completed/rejected which are handled above)
        // if ($oldStatus !== $newStatus && !in_array($newStatus, [WalletWithdrawalRequest::STATUS_COMPLETED, WalletWithdrawalRequest::STATUS_REJECTED])) {
        //     $withdrawalRequest->user->notify(new WithdrawalRequestStatusUpdatedNotification($withdrawalRequest));
        // }

        return redirect()->route('admin.wallet-withdrawal-requests.show', $withdrawalRequest)->with('success', 'Withdrawal request status updated successfully.');
    }
}

