<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\SubscriptionPlan; // Added
use App\Models\Subscription; // Added
use Spatie\Permission\Models\Role as SpatieRole; // Use an alias for Spatie's Role model
use App\Services\CreditService; // Added
use App\Services\WalletService; // Added
use Illuminate\Validation\Rules;
use Nwidart\Modules\Facades\Module;
use App\Models\ModulePermission;
use Illuminate\Support\Str; // Import the Str facade
use Illuminate\Support\Facades\Auth;

class AdminUserController extends Controller
{
    protected CreditService $creditService;
    protected WalletService $walletService;
    public function __construct(CreditService $creditService, WalletService $walletService)
    {
        // Middleware will be applied in the routes file now.
        // The constructor can be left empty or used for other dependency injections if needed.
        $this->creditService = $creditService;
        $this->walletService = $walletService;
    }

    public function index(Request $request)
    {
        $query = User::query();


        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%");
            });
        }

        if ($request->filled('role')) {
            $query->where('role', $request->input('role'));
        }

        if ($request->filled('status')) {
            $status = $request->input('status');
            if ($status === 'active') {
                $query->whereNull('suspended_at')->whereNotNull('email_verified_at');
            } elseif ($status === 'suspended') {
                $query->whereNotNull('suspended_at');
            } elseif ($status === 'unverified') {
                $query->whereNull('email_verified_at');
            }
        }

        $users = $query->latest()->paginate(15)->withQueryString();

        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $roles = []; // For the simple role column
        $assignableRoles = collect();

        if (Auth::user()->isSuperAdmin()) {
            $roles = ['user' => 'User', 'admin' => 'Admin', 'super_admin' => 'Super Admin'];
            $assignableRoles = SpatieRole::orderBy('name')->get();
        }
        return view('admin.users.create', compact('roles', 'assignableRoles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'string', 'in:user,admin,super_admin'],
            'spatie_roles' => 'nullable|array',
            'spatie_roles.*' => 'exists:roles,name',
        ]);

        $userData = [
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'is_admin' => in_array($request->role, ['admin', 'super_admin']),
            'email_verified_at' => now(),
        ];

        $user = User::create($userData);

        if ($request->has('spatie_roles') && Auth::user()->isSuperAdmin()) {
            if (config('permission.teams') === true) {
                $teamIdForRoles = $user->current_team_id ?? ($user->personalTeam ? $user->personalTeam->id : null);
                if (!$teamIdForRoles) {
                    return redirect()->back()->withInput()->with('error', 'Failed to assign roles: Team context could not be determined for team-specific roles.');
                }
                $user->setRoleTeamId($teamIdForRoles);
                $user->syncRoles($request->input('spatie_roles', []));
                $user->setRoleTeamId(null);
            } else {
                $user->syncRoles($request->input('spatie_roles', []));
            }
        } elseif (Auth::user()->isSuperAdmin()) {
            $user->syncRoles([]);
        }

        return redirect()->route('admin.users.index')->with('success', 'User created successfully.');
    }

    public function edit(User $user)
    {
        $roles = []; // For the simple role column
        $assignableRoles = collect();
        $userRoles = collect();

        if (Auth::user()->isSuperAdmin()) {
            $roles = ['user' => 'User', 'admin' => 'Admin', 'super_admin' => 'Super Admin'];
            $assignableRoles = SpatieRole::orderBy('name')->get();
            $userRoles = $user->getRoleNames();
        } elseif (Auth::user()->hasRole('admin')) {
            $roles = [$user->role => Str::title($user->role)];
        }

        $modules = Module::allEnabled();
        $assignedModules = [];
        $subscriptionPlans = SubscriptionPlan::where('active', true)->orderBy('name')->get();
        $currentSubscription = $user->subscriptions()->whereIn('status', ['active', 'trialing'])->orderBy('created_at', 'desc')->first();
        $assignedModules = ModulePermission::where('role_name', 'admin')->pluck('module_name')->toArray();

        return view('admin.users.edit', compact('user', 'roles', 'assignableRoles', 'userRoles', 'modules', 'assignedModules', 'subscriptionPlans', 'currentSubscription'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'role' => ['required', 'string', 'in:user,admin,super_admin'],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
            'spatie_roles' => 'nullable|array',
            'spatie_roles.*' => 'exists:roles,name',
        ]);

        $userData = [
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'is_admin' => in_array($request->role, ['admin', 'super_admin']),
        ];

        if (!Auth::user()->isSuperAdmin()) {
            unset($userData['role']);
            unset($userData['is_admin']);
        } elseif ($user->isSuperAdmin() && $request->role !== 'super_admin' && User::where('role', 'super_admin')->count() <= 1 && $user->id === Auth::id()) {
            return redirect()->back()->with('error', 'Cannot remove the last Super Admin role.')->withInput();
        }

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->password);
        }

        $user->update($userData);

        if (Auth::user()->isSuperAdmin()) {
            if (config('permission.teams') === true) {
                $teamIdForRoles = $user->current_team_id ?? ($user->personalTeam ? $user->personalTeam->id : null);
                if (!$teamIdForRoles) {
                    return redirect()->back()->withInput()->with('error', 'Failed to update roles: Team context could not be determined for team-specific roles.');
                }
                $user->setRoleTeamId($teamIdForRoles);
                $user->syncRoles($request->input('spatie_roles', []));
                $user->setRoleTeamId(null);
            } else {
                $user->syncRoles($request->input('spatie_roles', []));
            }
        }
        if (Auth::user()->isSuperAdmin() && $request->has('update_admin_role_modules')) { // Add a checkbox or similar to trigger this
            ModulePermission::where('role_name', 'admin')->delete(); // Clear existing for 'admin' role
            if ($request->has('modules')) {
                foreach ($request->modules as $moduleName) {
                    ModulePermission::create([
                        'role_name' => 'admin',
                        'module_name' => $moduleName
                    ]);
                }
            }
        }

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->id === Auth::id()) {
            return redirect()->route('admin.users.index')->with('error', 'You cannot delete your own account.');
        }
        if ($user->isSuperAdmin() && User::where('role', 'super_admin')->count() <= 1) {
             return redirect()->route('admin.users.index')->with('error', 'Cannot delete the last Super Admin.');
        }
        $user->delete();
        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully.');
    }

    public function suspend(User $user)
    {
        if ($user->id === Auth::id()) {
            return redirect()->route('admin.users.index')->with('error', 'You cannot suspend your own account.');
        }
         if ($user->isSuperAdmin() && !$user->suspended_at && User::where('role', 'super_admin')->whereNull('suspended_at')->count() <= 1) {
            return redirect()->route('admin.users.index')->with('error', 'Cannot suspend the last active Super Admin.');
        }
        $user->update(['suspended_at' => now()]);
        return redirect()->route('admin.users.index')->with('success', 'User suspended successfully.');
    }

    public function unsuspend(User $user)
    {
        $user->update(['suspended_at' => null]);
        return redirect()->route('admin.users.index')->with('success', 'User unsuspended successfully.');
    }

    public function updateUserSubscription(Request $request, User $user)
    {
        if (!Auth::user()->isSuperAdmin()) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Only Super Admins can manage user subscriptions.');
        }

        $request->validate([
            'subscription_plan_id' => 'required|exists:subscription_plans,id',
        ]);

        $newPlan = SubscriptionPlan::findOrFail($request->subscription_plan_id);

        $user->subscriptions()
            ->whereIn('status', ['active', 'trialing'])
            ->update([
                'status' => 'cancelled',
                'ends_at' => now(),
                'cancelled_at' => now()
            ]);

        // Create the new subscription assigned by admin
        $startDate = now();
        $endDate = null;
        if ($newPlan->interval === 'day') {
            $endDate = $startDate->copy()->addDays($newPlan->interval_count);
        } elseif ($newPlan->interval === 'week') {
            $endDate = $startDate->copy()->addWeeks($newPlan->interval_count);
        } elseif ($newPlan->interval === 'month') {
            $endDate = $startDate->copy()->addMonths($newPlan->interval_count);
        } elseif ($newPlan->interval === 'year') {
            $endDate = $startDate->copy()->addYears($newPlan->interval_count);
        }

        Subscription::create([
            'user_id' => $user->id,
            'subscription_plan_id' => $newPlan->id,
            'payment_gateway' => 'admin_assigned',
            'gateway_subscription_id' => 'ADMIN-' . strtoupper(Str::random(10)),
            'status' => 'active',
            'trial_ends_at' => null,
            'starts_at' => $startDate,
            'cancelled_at' => null,
        ]);

        return redirect()->route('admin.users.edit', $user)->with('success', 'Subscription plan assigned to user successfully.');
    }

    public function removeUserSubscription(Request $request, User $user)
    {
        if (!Auth::user()->isSuperAdmin()) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Only Super Admins can manage user subscriptions.');
        }

        $user->subscriptions()->whereIn('status', ['active', 'trialing'])->update(['status' => 'cancelled', 'ends_at' => now(), 'cancelled_at' => now()]);

        return redirect()->route('admin.users.edit', $user)->with('success', 'User subscription has been cancelled.');
    }

    public function adjustUserCredits(Request $request, User $user)
    {
        if (!Auth::user()->isSuperAdmin()) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Unauthorized action.');
        }

        $request->validate([
            'credit_adjustment_amount' => 'required|integer',
            'credit_adjustment_description' => 'required|string|max:255',
        ]);

        $amount = (int) $request->input('credit_adjustment_amount');
        $description = $request->input('credit_adjustment_description');

        try {
            if ($amount > 0) {
                $this->creditService->awardCredits($user, $amount, 'admin_adjustment', $description);
            } elseif ($amount < 0) {
                $this->creditService->spendCredits($user, abs($amount), 'admin_adjustment', $description);
            }
            return redirect()->route('admin.users.edit', $user)->with('success', 'User credit balance adjusted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Failed to adjust credit balance: ' . $e->getMessage());
        }
    }

    public function adjustUserWallet(Request $request, User $user)
    {
        if (!Auth::user()->isSuperAdmin()) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Unauthorized action.');
        }

        $request->validate([
            'wallet_adjustment_amount' => 'required|numeric',
            'wallet_adjustment_description' => 'required|string|max:255',
        ]);

        $amount = (float) $request->input('wallet_adjustment_amount');
        $description = $request->input('wallet_adjustment_description');

        try {
            $this->walletService->adjustBalance(
                $user,
                $amount,
                $description
            );
            return redirect()->route('admin.users.edit', $user)->with('success', 'User wallet balance adjusted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('admin.users.edit', $user)->with('error', 'Failed to adjust wallet balance: ' . $e->getMessage());
        }
    }

    public function loginAsUser(Request $request, User $user)
    {
        if (!function_exists('setting') || setting('magic_login_enabled', '0') != '1') {
            return redirect()->route('admin.users.index')->with('error', 'Magic login feature is currently disabled.');
        }

        $admin = Auth::user();

        // Only Super Admins can use this feature
        if (!$admin->isSuperAdmin()) {
            return redirect()->route('admin.users.index')->with('error', 'You are not authorized to perform this action.');
        }

        if ($admin->id === $user->id) {
            return redirect()->route('admin.users.index')->with('error', 'You cannot log in as yourself.');
        }

        session(['original_admin_id' => $admin->id]);
        session(['original_admin_name' => $admin->name]);
        session()->save();

        $this->guard()->logout();
        $this->guard()->login($user, true);

        if (Auth::guard('web')->check() && Auth::guard('web')->id() === $user->id) {
            // Auth check OK
        } else {
            if (session()->has('original_admin_id')) {
                $originalAdminId = session()->pull('original_admin_id');
                session()->forget('original_admin_name');
                Auth::guard('web')->loginUsingId($originalAdminId, true);
                return redirect()->route('admin.users.index')->with('error', 'Critical error: Failed to establish session for ' . $user->name . '. Your admin session has been restored.');
            }
            return redirect()->route('login')->with('error', 'Impersonation failed at user login step.');
        }

        $request->session()->regenerate();

        if (Auth::guard('web')->check() && Auth::guard('web')->id() === $user->id) {
            return redirect()->route('dashboard')->with('info', 'You are now logged in as ' . $user->name . '.');
        } else {
            if (session()->has('original_admin_id')) {
                $originalAdminId = session()->pull('original_admin_id');
                session()->forget('original_admin_name');
                Auth::guard('web')->loginUsingId($originalAdminId, true);
                return redirect()->route('admin.users.index')->with('error', 'Failed to maintain session for ' . $user->name . ' after regeneration. Your admin session has been restored.');
            }
            return redirect()->route('login')->with('error', 'Impersonation failed after session regeneration.');
        }
    }

    public function returnToAdmin(Request $request)
    {
        if (!session()->has('original_admin_id')) {
            return redirect('/');
        }

        $originalAdminId = session()->pull('original_admin_id');
        session()->forget('original_admin_name');
        session()->save();
        $this->guard()->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();
        Auth::loginUsingId($originalAdminId, true);
        $request->session()->regenerate();

        return redirect()->route('admin.dashboard')->with('success', 'Returned to your admin session.');
    }

    /**
     * Get the guard to be used during authentication.
     *
     * @return \Illuminate\Contracts\Auth\StatefulGuard
     */
    protected function guard()
    {
        return Auth::guard();
    }
}
