<?php

namespace App\Helpers;

use App\Models\FeatureRoleAccess;
use App\Models\User;
use Nwidart\Modules\Facades\Module;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Models\FeatureCreditCost;
use Illuminate\Contracts\Cache\Repository as CacheRepository;
use Nwidart\Modules\Module as NwidartModule;
use App\Models\ModulePermission;

class UserAccessHelper
{
    /**
     * Check if a user can access a specific module.
     * Considers if the module requires a subscription and if the user's plan includes it.
     */
    public static function canAccessModule(?User $user, string $moduleName): bool
    {
        if (!self::isBoilerplateActive()) {
            return false; // Boilerplate not active, deny access
        }

        $module = Module::find($moduleName);
        if (!$module || !$module->isEnabled()) {
            return false;
        }

        $subscriptionsEnabled = (function_exists('setting') && setting('subscriptions_enabled', '1') == '1');
        $moduleRequiresSubscriptionFromJson = $module->get('requires_subscription', true);

        if (!$subscriptionsEnabled) {
            return true;
        }

        if (!$moduleRequiresSubscriptionFromJson) {
            return true;
        }

        if (!$user) {
            return false;
        }

        $subscription = $user->currentSubscription();
        if (!$subscription) {
            return false;
        }

        if (!self::isPlanTargetedForUser($subscription->plan, $user)) {
            return false;
        }

        foreach ($subscription->plan->features as $planFeature) {
            $expectedModulePrefix = Str::slug($moduleName) . '_';
            if (Str::startsWith($planFeature['key'], $expectedModulePrefix) || $planFeature['key'] === 'access_module_' . $moduleName) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if a user can access a specific feature.
     * This is the core access control logic.
     */
    public static function canAccessFeature(?User $user, string $featureKey): bool
    {

        if (!self::isBoilerplateActive()) {
            return false;
        }

        $moduleSystemName = self::getModuleSystemNameFromFeatureKey($featureKey);
        $module = $moduleSystemName ? Module::find($moduleSystemName) : null;

        if ($moduleSystemName && (!$module || !$module->isEnabled())) {
            return false;
        }

        if ($moduleSystemName && $user) {
            $userRoleNames = $user->getRoleNames()->toArray();
            $moduleRequiresExplicitRolePermission = ModulePermission::where('module_name', $moduleSystemName)->exists();

            if ($moduleRequiresExplicitRolePermission) {
                 if (empty($userRoleNames)) {
                    return false;
                }
                $hasModulePermissionViaTable = ModulePermission::where('module_name', $moduleSystemName)
                                                              ->whereIn('role_name', $userRoleNames)
                                                              ->exists();
                if (!$hasModulePermissionViaTable) {
                    return false;
                }
            }
        }

        $featureDefinition = self::getFeatureDefinition($module, $featureKey);

        $subscriptionsEnabled = (function_exists('setting') && setting('subscriptions_enabled', '1') == '1');
        $creditsSystemEnabled = (function_exists('setting') && setting('credits_system_enabled', '0') == '1');
        $roleAccess = self::getRoleSpecificFeatureAccess($user, $featureKey);

        if ($roleAccess) {
            if ($roleAccess->access_type === 'denied') {
                return false;
            }
            if ($roleAccess->access_type === 'free') {
                if ($creditsSystemEnabled && isset($roleAccess->credit_cost) && $roleAccess->credit_cost > 0) {
                    if (!$user || $user->credit_balance < $roleAccess->credit_cost) {
                        return false;
                    }
                }
                return true;
            }
        }

        $baseModuleRequiresSub = $module ? $module->get('requires_subscription', true) : true;
        $baseFeatureRequiresSub = $featureDefinition ? ($featureDefinition['requires_subscription'] ?? $baseModuleRequiresSub) : $baseModuleRequiresSub;

        if ($creditsSystemEnabled && !$roleAccess) {
            $creditCost = self::getFeatureCreditCostValue($featureKey, $featureDefinition);
            if ($creditCost > 0) {
                if (!$user || $user->credit_balance < $creditCost) {
                    return false;
                }
                if (!$baseFeatureRequiresSub) {
                    return true;
                }
            }
        }
        if (!$roleAccess && !$baseFeatureRequiresSub && !($creditsSystemEnabled && self::getFeatureCreditCostValue($featureKey, $featureDefinition) > 0 && (!$user || $user->credit_balance < self::getFeatureCreditCostValue($featureKey, $featureDefinition)))) {
            return true;
        }

        $effectiveRequiresSubscription = ($roleAccess && $roleAccess->access_type === 'paid') || (!$roleAccess && $baseFeatureRequiresSub);

        if (!$effectiveRequiresSubscription) {
            return true;
        }

        if (!$subscriptionsEnabled) {
            return true;
        }

        if (!$moduleSystemName) {
             if (!$user) {
                return false;
             }
             $subscription = $user->currentSubscription();
             if (!$subscription || !self::isPlanTargetedForUser($subscription->plan, $user)) {
                return false;
             }
             foreach ($subscription->plan->features as $planFeature) {
                if ($planFeature['key'] === $featureKey) {
                    return true;
                }
             }
             return false;
        }

        if (!$user) {
            return false;
        }
        $subscription = $user->currentSubscription();

        if (!$subscription || !self::isPlanTargetedForUser($subscription->plan, $user)) {
            return false;
        }

        foreach ($subscription->plan->features as $planFeature) {
            if ($planFeature['key'] === $featureKey) {
                return true;
            }
        }
        return false;
    }

    private static function getFeatureCreditCostValue(string $featureKey, ?array $featureDefinition): int
    {
        $costFromDb = FeatureCreditCost::where('feature_key', $featureKey)->value('credit_cost');
        if ($costFromDb !== null) {
            return (int) $costFromDb;
        }
        if ($featureDefinition && isset($featureDefinition['credit_cost'])) {
            return (int) $featureDefinition['credit_cost'];
        }
        return 0;
    }

    /**
     * Get the usage limit for a specific feature for a user.
     * Returns null if unlimited or feature not accessible.
     * Returns 0 if explicitly set to 0 (which might mean unlimited depending on interpretation).
     */
    public static function getFeatureLimit(?User $user, string $featureKey): ?int
    {
        if (!self::canAccessFeature($user, $featureKey)) {
            return 0;
        }

        $subscriptionsEnabled = (function_exists('setting') && setting('subscriptions_enabled', '1') == '1');
        $roleAccess = self::getRoleSpecificFeatureAccess($user, $featureKey);

        if ($user && $subscriptionsEnabled) {
            $subscription = $user->currentSubscription();
            if ($subscription && self::isPlanTargetedForUser($subscription->plan, $user)) {
                foreach ($subscription->plan->features as $planFeature) {
                    if ($planFeature['key'] === $featureKey && isset($planFeature['limit'])) {
                        $limit = (int)$planFeature['limit'];
                        return $limit === 0 ? null : $limit;
                    }
                }
            }
        }

        if ($roleAccess && $roleAccess->access_type === 'free') {
            return is_null($roleAccess->default_limit) ? null : (int)$roleAccess->default_limit;
        }

        $moduleSystemName = self::getModuleSystemNameFromFeatureKey($featureKey);
        $module = $moduleSystemName ? Module::find($moduleSystemName) : null;
        $featureDefinition = null;
        if ($module) {
            $featureDefinition = self::getFeatureDefinition($module, $featureKey);
        }
        if ($featureDefinition && isset($featureDefinition['default_limit_if_free'])) {
            $defaultLimit = (int)$featureDefinition['default_limit_if_free'];
            return $defaultLimit === 0 ? null : $defaultLimit;
        }
        return null;
    }

    /**
     * Helper to get the canonical module system name from a feature key.
     */
    private static function getModuleSystemNameFromFeatureKey(string $featureKey): ?string
    {
        $parts = explode('_', $featureKey);
        if (count($parts) > 1) {
            $potentialModuleName = $parts[0];
            $module = Module::find($potentialModuleName);
            if ($module) {
                return $module->getName();
            }
        }
        return null;
    }

    /**
     * Helper to get the feature definition array from module.json.
     *
     * @param NwidartModule|null $module The Nwidart Module object or null.
     * @param string $featureKey The feature key to look for.
     * @return array|null The feature definition array or null if not found.
     */
    private static function getFeatureDefinition(?NwidartModule $module, string $featureKey): ?array
    {
        if (!$module) {
            return null;
        }

        $definedModuleFeatures = $module->get('features', []);

        foreach ($definedModuleFeatures as $def) {
            if (isset($def['key']) && $def['key'] === $featureKey) {
                return $def;
            }
        }
        return null;
    }

    /**
     * Helper to get role-specific access rules for a feature.
     * Considers multiple roles and precedence: denied > free > paid.
     */
    private static function getRoleSpecificFeatureAccess(?User $user, string $featureKey): ?FeatureRoleAccess
    {
        if (!$user) {
            return null;
        }
        $userRoleIds = $user->roles->pluck('id')->toArray();
        if (empty($userRoleIds)) {
            return null;
        }

        $accessRules = FeatureRoleAccess::where('feature_key', $featureKey)
                                        ->whereIn('role_id', $userRoleIds)
                                        ->get();

        if ($accessRules->isEmpty()) {
            return null;
        }

        if ($deniedRule = $accessRules->firstWhere('access_type', 'denied')) {
            $deniedRule->credit_cost = null;
            $deniedRule->default_limit = null;
            return $deniedRule;
        }
        if ($freeRule = $accessRules->firstWhere('access_type', 'free')) {
            return $freeRule;
        }
        if ($paidRule = $accessRules->firstWhere('access_type', 'paid')) {
            return $paidRule;
        }
        return null;
    }

    /**
     * Helper to check if a subscription plan is targeted for the user's role(s).
     */
    public static function isPlanTargetedForUser(\App\Models\SubscriptionPlan $plan, User $user): bool
    {
        if (empty($plan->target_role)) {
            return true;
        }
        return $user->hasRole($plan->target_role);
    }

    /**
     * Blade directive helper: @canAccessFeature('feature_key')
     */
    public static function bladeCanAccessFeature(string $featureKey): bool
    {
        return static::canAccessFeature(Auth::user(), $featureKey);
    }

    /**
     * Checks if the boilerplate is properly activated by validating its token.
     * Caches the validation result.
     */
    private static function isBoilerplateActive(): bool
    {
        return true;
    }
}
