<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Log;

if (!function_exists('active_public_route_url')) {
    /**
     * Get the URL for a public-facing route, preferring LandingPage module if active.
     *
     * @param string $baseRouteName The base name of the route (e.g., 'tenders.index', 'companies.show').
     * @param array $parameters Route parameters.
     * @return string The generated URL, or '#' as a fallback.
     */
    function active_public_route_url(string $baseRouteName, array $parameters = []): string
    {
        $landingPageEnabled = function_exists('setting') && setting('landing_page_enabled', '0') == '1';

        $landingPageRouteName = 'landingpage.' . $baseRouteName;
        $businessDirectoryRouteName = 'frontend.businessdirectory.' . $baseRouteName;

        if ($landingPageEnabled && Route::has($landingPageRouteName)) {
            return route($landingPageRouteName, $parameters);
        }

        if (Route::has($businessDirectoryRouteName)) {
            return route($businessDirectoryRouteName, $parameters);
        }

        Log::warning("Public route URL for '{$baseRouteName}' not found for either LandingPage or BusinessDirectory modules.");
        return '#';
    }
}

if (!function_exists('get_active_public_route_name')) {
    /**
     * Get the resolved public-facing route name, preferring LandingPage module if active.
     *
     * @param string $baseRouteName The base name of the route (e.g., 'tenders.index', 'companies.show').
     * @return string|null The resolved route name, or null if not found.
     */
    function get_active_public_route_name(string $baseRouteName): ?string
    {
        $landingPageEnabled = function_exists('setting') && setting('landing_page_enabled', '0') == '1';

        $landingPageRouteName = 'landingpage.' . $baseRouteName;
        $businessDirectoryRouteName = 'frontend.businessdirectory.' . $baseRouteName;

        if ($landingPageEnabled && Route::has($landingPageRouteName)) {
            return $landingPageRouteName;
        }

        if (Route::has($businessDirectoryRouteName)) {
            return $businessDirectoryRouteName;
        }

        return null;
    }
}
