<?php

namespace App\Actions\Jetstream;

use App\Models\Team;
use App\Models\User;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Laravel\Jetstream\Contracts\CreatesTeams;
use Laravel\Jetstream\Events\AddingTeam;
use Laravel\Jetstream\Jetstream;
use Illuminate\Support\Facades\Log; // Import the Log Facade

class CreateTeam implements CreatesTeams
{
    /**
     * Validate and create a new team for the given user.
     *
     * @param  array<string, string>  $input
     * @return \App\Models\Team|null
     */
    public function create(User $user, array $input): ?Team
    {
        Gate::forUser($user)->authorize('create', Jetstream::newTeamModel());
        Log::info("[CreateTeam Action] - User {$user->id} attempting to create team."); // Log attempt

        try {
            Validator::make($input, [
                'name' => ['required', 'string', 'max:255'],
            ])->validateWithBag('createTeam'); // This will throw ValidationException on failure
            Log::info("[CreateTeam Action] - User {$user->id} - Validation passed for team name: {$input['name']}"); // Log successful validation

            AddingTeam::dispatch($user);
            Log::info("[CreateTeam Action] - User {$user->id} - AddingTeam event dispatched."); // Log event dispatch

            $team = $user->ownedTeams()->create([
                'name' => $input['name'],
                'personal_team' => false,
            ]);
            Log::info("[CreateTeam Action] - User {$user->id} - Team '{$team->name}' (ID: {$team->id}) created successfully."); // Log successful creation

            $user->switchTeam($team);
            Log::info("[CreateTeam Action] - User {$user->id} - Switched to new team '{$team->name}' (ID: {$team->id})."); // Log successful team switch

            return $team; // Return the created team
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::warning("[CreateTeam Action] - User {$user->id} - Validation failed: " . json_encode($e->errors())); // Log validation errors
            throw $e; // Re-throw the validation exception so Livewire can handle it
        } catch (\Exception $e) { // Catch any other generic exceptions
            Log::error("[CreateTeam Action] - User {$user->id} - An unexpected error occurred: " . $e->getMessage(), ['exception' => $e]);
            throw $e; // Re-throw to make it visible and let Livewire handle if possible
        }
    }
}
