<?php

namespace App\Actions\Fortify;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule; // Import Rule for exists validation
use Laravel\Fortify\Contracts\CreatesNewUsers;
use Laravel\Jetstream\Jetstream;
use Spatie\Permission\Models\Role as SpatieRole; // Import Spatie Role
use Illuminate\Support\Facades\DB; // Import DB for transaction

class CreateNewUser implements CreatesNewUsers
{
    use PasswordValidationRules;

    /**
     * Validate and create a newly registered user.
     *
     * @param  array<string, string>  $input
     */
    public function create(array $input): User
    {
        $enableRoleSelection = function_exists('setting') && setting('enable_role_selection_on_registration', '0') == '1';

        $validationRules = [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => $this->passwordRules(),
            'terms' => Jetstream::hasTermsAndPrivacyPolicyFeature() ? ['accepted', 'required'] : '',
        ];

        if ($enableRoleSelection) {
            $validationRules['selected_role'] = [
                'nullable', // Allow it to be empty if user doesn't pick or no roles are shown
                'string',
                Rule::exists('roles', 'name')->where(function ($query) {
                    // Ensure the role exists, is for the 'web' guard, and is marked for registration
                    return $query->where('show_on_registration', true)->where('guard_name', 'web');
                }),
            ];
        }

        Validator::make($input, $validationRules)->validate();

        return DB::transaction(function () use ($input, $enableRoleSelection) {
            return tap(User::create([
                'name' => $input['name'],
                'email' => $input['email'],
                'password' => Hash::make($input['password']),
                'role' => 'user', // Default simple role for new users
                'is_admin' => false, // New users are not admins by default
                'email_verified_at' => (config('fortify.features.email-verification', false)) ? null : now(), // Set based on Fortify config
            ]), function (User $user) use ($input, $enableRoleSelection) {
                if (method_exists($this, 'createTeam')) { // Check if createTeam method exists (Jetstream specific)
                    $this->createTeam($user);
                }

                // Assign Spatie role
                $assignedSpatieRoleName = null;
                if ($enableRoleSelection && !empty($input['selected_role'])) {
                    $selectedSpatieRole = SpatieRole::where('name', $input['selected_role'])
                                                    ->where('show_on_registration', true)
                                                    ->where('guard_name', 'web')
                                                    ->first();
                    if ($selectedSpatieRole) {
                        $assignedSpatieRoleName = $selectedSpatieRole->name;
                    }
                }

                $user->assignRole($assignedSpatieRoleName ?? 'user'); // Assign selected or default 'user' Spatie role
            });
        });
    }
}
