<?php

namespace Modules\WebPilotAI\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Modules\WebPilotAI\Models\Website;
use App\Models\User; // Assuming your User model is in App\Models
use Modules\WebPilotAI\Models\AIModel;

class UserSiteController extends Controller
{
    /**
     * Display a listing of all user websites.
     */
    public function index()
    {
        $websites = Website::with(['user', 'aiModel']) // Eager load relationships
                            ->orderBy('created_at', 'desc')
                            ->paginate(15);
        return view('webpilotai::admin.usersites.index', compact('websites'));
    }

    /**
     * Show the form for creating a new website (by admin).
     */
    public function create()
    {
        $users = User::orderBy('name')->get();
        $aiModels = AIModel::where('is_active', true)->orderBy('name')->get();
        return view('webpilotai::admin.usersites.create', compact('users', 'aiModels'));
    }

    /**
     * Store a newly created website in storage (by admin).
     */
    public function store(Request $request): RedirectResponse
    {
        $validatedData = $request->validate([
            'user_id' => 'required|exists:users,id',
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:10',
            'ai_model_id' => 'required|exists:ai_models,id',
            'status' => 'required|string', // Admin can set status directly
        ]);

        Website::create($validatedData);

        return redirect()->route('admin.webpilotai.user-sites.index')->with('success', 'User website created successfully.');
    }

    /**
     * Display the specified website.
     */
    public function show(Website $user_site) // Route model binding (Laravel 7+ uses singular for resource param)
    {
        $user_site->load(['user', 'aiModel']);
        return view('webpilotai::admin.usersites.show', ['website' => $user_site]);
    }

    /**
     * Show the form for editing the specified website.
     */
    public function edit(Website $user_site)
    {
        $users = User::orderBy('name')->get();
        $aiModels = AIModel::orderBy('name')->get(); // Get all models, admin might assign inactive one
        return view('webpilotai::admin.usersites.edit', ['website' => $user_site, 'users' => $users, 'aiModels' => $aiModels]);
    }

    /**
     * Update the specified website in storage.
     */
    public function update(Request $request, Website $user_site): RedirectResponse
    {
        $validatedData = $request->validate([
            'user_id' => 'required|exists:users,id',
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:10',
            'ai_model_id' => 'required|exists:ai_models,id',
            'status' => 'required|string',
            'generated_content_path' => 'nullable|string',
            'generation_error' => 'nullable|string',
        ]);

        $user_site->update($validatedData);

        return redirect()->route('admin.webpilotai.user-sites.index')->with('success', 'User website updated successfully.');
    }

    // destroy method can be added if admin should be able to delete sites
    // public function destroy(Website $user_site) { ... }
}