<?php

namespace Modules\WebPilotAI\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\WebPilotAI\Models\AIModel; // Import the AIModel

class AIModelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Fetch models from the database, ordered by name
        $models = AIModel::orderBy('name')->get();

        return view('webpilotai::admin.aimodels.index', compact('models'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('webpilotai::admin.aimodels.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request) {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            // The 'unique' rule will now work as the table and model are set up.
            'identifier' => 'required|string|max:255|unique:ai_models,identifier',
            'provider' => 'required|string|max:100',
            'api_key_setting_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_premium' => 'nullable|boolean',
        ]);

        // Adjust 'is_active' and 'is_premium' as checkboxes send '1' or nothing
        $validatedData['is_active'] = $request->has('is_active');
        $validatedData['is_premium'] = $request->has('is_premium');

        // Create the AIModel record in the database
        AIModel::create($validatedData);

        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model added successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \Modules\WebPilotAI\Models\AIModel  $model  // Route model binding
     * @return \Illuminate\Http\Response
     */
    public function edit(AIModel $model) // Using Route Model Binding
    {
        return view('webpilotai::admin.aimodels.edit', compact('model'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Modules\WebPilotAI\Models\AIModel  $model // Route model binding
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, AIModel $model) // Using Route Model Binding
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            // Ensure identifier is unique, ignoring the current model's identifier
            'identifier' => 'required|string|max:255|unique:ai_models,identifier,' . $model->id,
            'provider' => 'required|string|max:100',
            'api_key_setting_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_premium' => 'nullable|boolean',
        ]);

        $validatedData['is_active'] = $request->has('is_active');
        $validatedData['is_premium'] = $request->has('is_premium');

        $model->update($validatedData);

        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \Modules\WebPilotAI\Models\AIModel  $model // Route model binding
     * @return \Illuminate\Http\Response
     */
    public function destroy(AIModel $model) // Using Route Model Binding
    {
        $model->delete();
        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model deleted successfully.');
    }
}
