<?php

namespace Modules\WebPilotAI\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\JsonResponse; // Import JsonResponse
use Illuminate\Validation\Rule;
use Modules\WebPilotAI\Models\AIModel;
use Modules\WebPilotAI\Models\WebsiteAsset; // Import WebsiteAsset model
use Illuminate\Support\Str; // Import Str for slug generation
use Modules\WebPilotAI\Models\AITemplate; // Import AITemplate model
use Modules\WebPilotAI\Models\AIStylePreset; // Import AIStylePreset model
use Modules\WebPilotAI\Models\Website;
use App\Models\Setting; // Assuming your Setting model is here
use Modules\WebPilotAI\Jobs\ProcessWebsiteGeneration;
use Illuminate\Support\Facades\Storage; // For deleting generated content
use Illuminate\Support\Facades\Log; // For logging asset deletions
use Illuminate\Support\Facades\Crypt; // For password encryption/decryption

class WebsiteController extends Controller
{
    /**
     * Display a listing of the user's websites.
     */
    public function index()
    {
        $websites = Website::where('user_id', Auth::id())
                            ->with('aiModel') // Eager load the AI model relationship
                            ->orderBy('created_at', 'desc')
                            ->paginate(10); // Paginate results

        return view('webpilotai::user.websites.index', compact('websites'));
    }

    /**
     * Show the form for creating a new website.
     */
    public function create()
    {
        $user = Auth::user();
        $premiumAccessFeatureKey = 'webpilotai_premium_items_access';
        $websiteCreationFeatureKey = 'webpilotai_generate_website';
        $limitReached = false;
        $maxWebsites = null;
        $canAccessPremium = false;

        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $premiumAccessFeatureKey) {
                    $canAccessPremium = true; // If the key exists, access is granted
                }
                if (isset($feature['key']) && $feature['key'] === $websiteCreationFeatureKey && isset($feature['limit'])) {
                    $maxWebsites = (int) $feature['limit'];
                    $currentWebsiteCount = Website::where('user_id', $user->id)->count();
                    if ($currentWebsiteCount >= $maxWebsites) {
                        $limitReached = true;
                    }
                    // No break here if premiumAccessFeatureKey might appear later in the array
                }
            }
        }
        // If $maxWebsites is still null, it means the feature might not be limited or not part of the plan.

        $aiModelsQuery = AIModel::where('is_active', true);
        if (!$canAccessPremium) {
            $aiModelsQuery->where('is_premium', false);
        }
        $aiModels = $aiModelsQuery->orderBy('provider')->orderBy('name')->get();

        $aiTemplatesQuery = AITemplate::where('is_active', true);
        if (!$canAccessPremium) {
            $aiTemplatesQuery->where('is_premium', false);
        }
        $aiTemplates = $aiTemplatesQuery
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $aiStylePresetsQuery = AIStylePreset::where('is_active', true);
        if (!$canAccessPremium) {
            $aiStylePresetsQuery->where('is_premium', false);
        }
        $aiStylePresets = $aiStylePresetsQuery
            ->orderBy('type') // Group by type visually if needed in view, or sort by name
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $enableFreeTier = Setting::getValue('webpilotai.enable_free_tier_model', false);
        $freeTierModelIdentifier = null;
        if ($enableFreeTier) {
            $freeTierModelIdentifier = Setting::getValue('webpilotai.free_tier_model_name');
            // Ensure the free tier model is always available if enabled, regardless of premium status filtering
            if ($freeTierModelIdentifier && $aiModels->where('identifier', $freeTierModelIdentifier)->isEmpty()) {
                $freeModel = AIModel::where('identifier', $freeTierModelIdentifier)->where('is_active', true)->first();
                if ($freeModel) $aiModels->prepend($freeModel); // Add it if not already in the filtered list
            }
        }

        return view('webpilotai::user.websites.create', compact('aiModels', 'aiTemplates', 'aiStylePresets', 'freeTierModelIdentifier', 'limitReached', 'maxWebsites'));
    }

    /**
     * Store a newly created website in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = Auth::user();
        $premiumAccessFeatureKey = 'webpilotai_premium_items_access';
        $websiteCreationFeatureKey = 'webpilotai_generate_website';

        // Check "Max AI Websites" limit before proceeding
        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            $maxWebsites = null;
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $websiteCreationFeatureKey && isset($feature['limit'])) {
                    $maxWebsites = (int) $feature['limit'];
                    break;
                }
            }
            if (!is_null($maxWebsites)) {
                $currentWebsiteCount = Website::where('user_id', $user->id)->count();
                if ($currentWebsiteCount >= $maxWebsites) {
                    return redirect()->route('frontend.webpilotai.websites.create')
                                     ->with('error', "You have reached your limit of {$maxWebsites} websites for your current plan.");
                }
            }

            // --- Credit System Integration for Website Generation ---
            $websiteGenerationCost = 0; // Default to 0
            if ($user->currentSubscription() && $user->currentSubscription()->plan) {
                $cost = $user->currentSubscription()->plan->getFeatureCreditCost($websiteCreationFeatureKey);
                if (!is_null($cost)) {
                    $websiteGenerationCost = (int) $cost;
                } else {
                    Log::warning("Credit cost for feature '{$websiteCreationFeatureKey}' is not defined in FeatureCreditCost table. Assuming 0 cost for user {$user->id} during website creation.");
                }
            } else {
                Log::warning("User {$user->id} does not have an active subscription plan or plan details are missing. Credit cost check for '{$websiteCreationFeatureKey}' resulted in 0 cost assumption during website creation.");
            }

            $userCreditBalance = $user->current_credit_balance; 

            // Check credits only if the feature has a cost
            if ($websiteGenerationCost > 0 && $userCreditBalance < $websiteGenerationCost) { 
                return redirect()->route('frontend.webpilotai.websites.create')
                                 ->with('error', "Insufficient credits. You need at least {$websiteGenerationCost} credits to generate a new website.");
            }
            // --- End Credit System Integration ---

        }
        
        $canAccessPremium = false;
        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $premiumAccessFeatureKey) {
                    $canAccessPremium = true;
                    break;
                }
            }
        }
        
        $enableFreeTier = Setting::getValue('webpilotai.enable_free_tier_model', false);
        $freeTierModelIdentifier = null;
        if ($enableFreeTier) {
            $freeTierModelIdentifier = Setting::getValue('webpilotai.free_tier_model_name');
        }

        $validatedData = $request->validate([
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:50', // Min length for a decent prompt
            'ai_model_id' => [
                'required',
                Rule::exists('ai_models', 'id')->where(function ($query) use ($canAccessPremium, $freeTierModelIdentifier) {
                    $query->where('is_active', true)
                          ->where('api_key_setting_name', '!=', null) // check it has an api key setting name

                          ->where(function ($subQuery) use ($canAccessPremium, $freeTierModelIdentifier) {
                              $subQuery->where('is_premium', false) // Allow non-premium
                                       ->orWhere(function ($premiumQuery) use ($canAccessPremium) { // Or if premium, user must have access
                                           $premiumQuery->where('is_premium', true)
                                                        ->whereRaw('?', [$canAccessPremium]); // Use whereRaw to pass boolean
                                       })
                                       ->when($freeTierModelIdentifier, function ($q, $identifier) { // Or if it's the free tier model
                                            $q->orWhere('identifier', $identifier);
                                       });
                          });
                }),
            ],
            'ai_template_id' => ['nullable', Rule::exists('ai_templates', 'id')->where(function ($query) use ($canAccessPremium) {
                $query->where('is_active', true)
                      ->where(function ($subQuery) use ($canAccessPremium) {
                          $subQuery->where('is_premium', false)
                                   ->orWhere(function ($premiumQuery) use ($canAccessPremium) {
                                       $premiumQuery->where('is_premium', true)
                                                    ->whereRaw('?', [$canAccessPremium]);
                                   });
                      });
            })],
            'ai_style_preset_ids' => 'nullable|array',
            'ai_style_preset_ids.*' => ['nullable', Rule::exists('ai_style_presets', 'id')->where(function ($query) use ($canAccessPremium) {
                $query->where('is_active', true)
                      ->where(function ($subQuery) use ($canAccessPremium) {
                          $subQuery->where('is_premium', false)
                                   ->orWhere(function ($premiumQuery) use ($canAccessPremium) {
                                       $premiumQuery->where('is_premium', true)
                                                    ->whereRaw('?', [$canAccessPremium]);
                                   });
                      });
            })],
            'pages' => 'nullable|array', // Expect an array of page names
            'pages.*' => 'nullable|string|max:100', // Each page name is a string
            'assets' => 'nullable|array', // Expect an array of files for assets
            'assets.*' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:5120', // Validate each file: image, common types, max 5MB
        ]);

        $website = Website::create([
            'user_id' => Auth::id(),
            'name' => $validatedData['name'] ?? 'My AI Website - ' . now()->format('Y-m-d H:i'),
            'description_prompt' => $validatedData['description_prompt'],
            'ai_model_id' => $validatedData['ai_model_id'],
            'ai_template_id' => $validatedData['ai_template_id'] ?? null,
            'pages_structure' => $this->formatPagesStructure($request->input('pages')),
            'status' => 'pending_generation', // Initial status
        ]);


        // Sync AI Style Presets
        if ($request->has('ai_style_preset_ids')) {
            $website->aiStylePresets()->sync($validatedData['ai_style_preset_ids']);
        }

        // Handle Asset Uploads
        if ($request->hasFile('assets')) {
            $userId = Auth::id();
            foreach ($request->file('assets') as $file) {
                if ($file && $file->isValid()) {
                    $path = $file->store("webpilotai/user_uploads/{$userId}/assets", 'public'); // Store on 'public' disk
                    $website->assets()->create([
                        'user_id' => $userId,
                        'type' => 'image', // Assuming all uploads here are images for now
                        'disk' => 'public',
                        'path' => $path,
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }
        }

        // Deduct credits for website generation
        if ($websiteGenerationCost > 0) {
            $transactionType = 'spend_feature_webpilotai_generate_website';
            $description = "AI Website Generation: " . ($website->name ?: 'Untitled Website');
            if (!$user->deductCredits($websiteGenerationCost, $transactionType, $description, $website)) {
                // If deduction fails after creating the website record, this is problematic.
                // You might want to delete the created $website record or mark it as pending payment.
                Log::critical("Failed to deduct credits for user {$user->id} for website ID {$website->id} AFTER website record creation. Cost: {$websiteGenerationCost}. Manual intervention may be required. Website creation will proceed but credits were not deducted.");
                // For now, we'll proceed but this needs robust handling.
            }
        }

        // Dispatch the job to handle AI generation
        ProcessWebsiteGeneration::dispatch($website);

        return redirect()->route('frontend.webpilotai.websites.index')->with('success', 'Your website request has been submitted! Generation will begin shortly.');
    }

    /**
     * Display the specified website.
     */
    public function show(Website $website) // Route model binding
    {
        // Ensure the authenticated user owns this website
        $this->authorize('view', $website); // Requires a WebsitePolicy
        $website->load(['aiModel', 'aiTemplate', 'aiStylePresets', 'assets']); // Eager load assets
        return view('webpilotai::user.websites.show', compact('website'));
    }

    /**
     * Check the status of the specified website via AJAX.
     */
    public function checkStatus(Website $website): JsonResponse
    {
        // Ensure the authenticated user owns this website
        $this->authorize('view', $website); // Requires a WebsitePolicy

        // Reload the website to get the latest status from the database
        $website->refresh();

        // Prepare data for the JSON response
        $data = [
            'id' => $website->id,
            'status' => $website->status,
            'status_friendly' => Str::title(str_replace('_', ' ', $website->status)),
            'generation_error' => $website->generation_error,
            'last_generated_at' => $website->last_generated_at ? $website->last_generated_at->format('M d, Y H:i A') : 'Never',
            'deployment_details' => $website->deployment_details,
            'generated_content_path' => $website->generated_content_path ? asset($website->generated_content_path) : null,
            'can_download' => $website->status === Website::STATUS_COMPLETED && $website->generated_content_path !== null,
            'can_deploy_form' => $website->status === Website::STATUS_COMPLETED && $website->generated_content_path !== null,
            'name' => $website->name ?: 'Untitled Website', // Include name for notification
        ];

        return response()->json($data);
    }

    /**
     * Show the form for editing the specified website.
     */
    public function edit(Website $website) // Route model binding
    {
        $this->authorize('update', $website); // Requires a WebsitePolicy
        $user = Auth::user();
        $premiumAccessFeatureKey = 'webpilotai_premium_items_access';
        $canAccessPremium = false;

        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $premiumAccessFeatureKey) {
                    $canAccessPremium = true;
                    break;
                }
            }
        }

        $aiModelsQuery = AIModel::where('is_active', true);
        if (!$canAccessPremium) {
            $aiModelsQuery->where('is_premium', false);
        }
        $aiModels = $aiModelsQuery->orderBy('provider')->orderBy('name')->get();

        $aiTemplatesQuery = AITemplate::where('is_active', true);
        if (!$canAccessPremium) {
            $aiTemplatesQuery->where('is_premium', false);
        }
        $aiTemplates = $aiTemplatesQuery
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $aiStylePresetsQuery = AIStylePreset::where('is_active', true);
        if (!$canAccessPremium) {
            $aiStylePresetsQuery->where('is_premium', false);
        }
        $aiStylePresets = $aiStylePresetsQuery
            ->orderBy('type')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $enableFreeTier = Setting::getValue('webpilotai.enable_free_tier_model', false);
        $freeTierModelIdentifier = null;
        if ($enableFreeTier) {
            $freeTierModelIdentifier = Setting::getValue('webpilotai.free_tier_model_name');
            // Ensure the free tier model is always available if enabled, regardless of premium status filtering
            // And also ensure the currently selected model for the website is available in the dropdown
            $currentModelId = $website->ai_model_id;
            if ($freeTierModelIdentifier && $aiModels->where('identifier', $freeTierModelIdentifier)->isEmpty()) {
                $freeModel = AIModel::where('identifier', $freeTierModelIdentifier)->where('is_active', true)->first();
                if ($freeModel) $aiModels->prepend($freeModel);
            }
            if ($currentModelId && $aiModels->where('id', $currentModelId)->isEmpty()) {
                $currentSelectedModel = AIModel::where('id', $currentModelId)->where('is_active', true)->first();
                if ($currentSelectedModel) $aiModels->prepend($currentSelectedModel);
            }
            $aiModels = $aiModels->unique('id')->sortBy('name')->sortBy('provider'); // Re-sort after potential prepends
        }

        return view('webpilotai::user.websites.edit', compact('website', 'aiModels', 'aiTemplates', 'aiStylePresets', 'freeTierModelIdentifier'));
    }

    /**
     * Update the specified website in storage.
     */
    public function update(Request $request, Website $website): RedirectResponse
    {
        $this->authorize('update', $website);
        $user = Auth::user();
        $premiumAccessFeatureKey = 'webpilotai_premium_items_access';
        $canAccessPremium = false;
        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $premiumAccessFeatureKey) {
                    $canAccessPremium = true;
                    break;
                }
            }
        }
        
        $enableFreeTier = Setting::getValue('webpilotai.enable_free_tier_model', false);
        $freeTierModelIdentifier = null;
        if ($enableFreeTier) {
            $freeTierModelIdentifier = Setting::getValue('webpilotai.free_tier_model_name');
        }

        $validatedData = $request->validate([
            'name' => 'nullable|string|max:255',
            'description_prompt' => 'required|string|min:50',
            'ai_model_id' => [
                'required',
                Rule::exists('ai_models', 'id')->where(function ($query) use ($canAccessPremium, $freeTierModelIdentifier, $website) {
                    $query->where('is_active', true)
                          ->where(function ($subQuery) use ($canAccessPremium, $freeTierModelIdentifier, $website) {
                              $subQuery->where('is_premium', false) // Allow non-premium
                                       ->orWhere(function ($premiumQuery) use ($canAccessPremium) { // Or if premium, user must have access
                                           $premiumQuery->where('is_premium', true)
                                                        ->whereRaw('?', [$canAccessPremium]);
                                       })
                                       ->when($freeTierModelIdentifier, function ($q, $identifier) { // Or if it's the free tier model
                                            $q->orWhere('identifier', $identifier);
                                       })
                                       ->orWhere('id', $website->ai_model_id); // Or if it's the currently selected model (even if premium and no access)
                          });
                }),
            ],
            'ai_template_id' => ['nullable', Rule::exists('ai_templates', 'id')->where(function ($query) use ($canAccessPremium, $website) {
                $query->where('is_active', true)
                      ->where(function ($subQuery) use ($canAccessPremium, $website) {
                          $subQuery->where('is_premium', false)
                                   ->orWhere(function ($premiumQuery) use ($canAccessPremium) {
                                       $premiumQuery->where('is_premium', true)
                                                    ->whereRaw('?', [$canAccessPremium]);
                                   })
                                   ->orWhere('id', $website->ai_template_id); // Allow if it's the currently selected template
                      });
            })],
            'ai_style_preset_ids' => 'nullable|array',
            'ai_style_preset_ids.*' => ['nullable', Rule::exists('ai_style_presets', 'id')->where(function ($query) use ($canAccessPremium) {
                $query->where('is_active', true)
                      ->where(function ($subQuery) use ($canAccessPremium) {
                          $subQuery->where('is_premium', false)
                                   ->orWhere(function ($premiumQuery) use ($canAccessPremium) {
                                       $premiumQuery->where('is_premium', true)
                                                    ->whereRaw('?', [$canAccessPremium]);
                                   });
                      });
            })],
            'assets' => 'nullable|array', // New assets being uploaded
            'assets.*' => 'nullable|image|mimes:jpg,jpeg,png,gif,webp|max:5120', // Validate new files
            'pages' => 'nullable|array',
            'pages.*' => 'nullable|string|max:100',
            'asset_descriptions' => 'nullable|array', // Expect an array of descriptions for existing assets
            'asset_descriptions.*' => 'nullable|string|max:500', // Validate each description
            'regenerate' => 'nullable|boolean', // For the re-generate checkbox
        ]);

        $website->name = $validatedData['name'] ?? $website->name; // Keep old name if new one is not provided
        $website->description_prompt = $validatedData['description_prompt'];
        $website->ai_model_id = $validatedData['ai_model_id'];
        $website->ai_template_id = $validatedData['ai_template_id'] ?? null;
        $website->pages_structure = $this->formatPagesStructure($request->input('pages'));

        // Only update status and clear error if regeneration is requested
        if ($request->boolean('regenerate')) { // Use boolean() for checkbox
            $website->status = Website::STATUS_PENDING_REGENERATION;
            $website->generation_error = null; // Clear previous errors if re-generating
        }
        $website->save(); // Save basic details first

        // Sync AI Style Presets
        $website->aiStylePresets()->sync($request->input('ai_style_preset_ids', []));

        // Handle New Asset Uploads (similar to store)
        if ($request->hasFile('assets')) {
            $userId = Auth::id();
            foreach ($request->file('assets') as $file) {
                if ($file && $file->isValid()) {
                    $path = $file->store("webpilotai/user_uploads/{$userId}/assets", 'public');
                    $website->assets()->create([
                        'user_id' => $userId,
                        'type' => 'image',
                        'disk' => 'public',
                        'path' => $path,
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }
        }

        // Handle Deletion of Existing Assets
        if ($request->has('remove_assets')) {
            $assetIdsToRemove = $request->input('remove_assets');
            $assetsToDelete = $website->assets()
                                      ->whereIn('id', $assetIdsToRemove)
                                      ->where('user_id', Auth::id()) // Security check
                                      ->get();

            foreach ($assetsToDelete as $asset) {
                if (Storage::disk($asset->disk)->exists($asset->path)) {
                    Storage::disk($asset->disk)->delete($asset->path);
                    Log::info("Deleted asset file from storage: {$asset->path} for website ID: {$website->id}");
                }
                $asset->delete();
                Log::info("Deleted WebsiteAsset record ID: {$asset->id} for website ID: {$website->id}");
            }
        }

        // Now, update descriptions for assets that were NOT removed
        $assetDescriptions = $request->input('asset_descriptions', []);
        $website->load('assets'); // Reload the relationship to get the current state
        foreach ($website->assets as $asset) {
            if (array_key_exists($asset->id, $assetDescriptions) && $asset->description !== $assetDescriptions[$asset->id]) {
                 $asset->update(['description' => $assetDescriptions[$asset->id]]);
                 Log::info("Updated description for WebsiteAsset ID: {$asset->id} for website ID: {$website->id}");
            }
        }

        $message = 'Website details updated successfully.';
        if ($request->boolean('regenerate')) { // Use boolean() for checkbox
            ProcessWebsiteGeneration::dispatch($website);
            $message = 'Website details updated. Regeneration process has started.';
        }

        return redirect()->route('frontend.webpilotai.websites.show', $website->id)->with('success', $message);
    }

    /**
     * Remove the specified website from storage.
     */
    public function destroy(Website $website): RedirectResponse
    {
        $this->authorize('delete', $website);

        if ($website->generated_content_path) {
            $storagePath = str_replace('storage/', 'public/', $website->generated_content_path);
             if (Storage::disk('local')->exists($storagePath)) {
                 Storage::disk('local')->delete($storagePath);
                 Log::info("Deleted generated ZIP file from storage: {$storagePath} for website ID: {$website->id}");
             }
        }

        $website->assets()->each(function ($asset) use ($website) {
            if (Storage::disk($asset->disk)->exists($asset->path)) {
                Storage::disk($asset->disk)->delete($asset->path);
                Log::info("Deleted asset file from storage during website deletion: {$asset->path} for website ID: {$website->id}");
            }
        });

        $website->delete();
        return redirect()->route('frontend.webpilotai.websites.index')->with('success', 'Website deleted successfully.');
    }

    /**
     * Show the form for deploying the website.
     */
    public function showDeployForm(Website $website)
    {
        $this->authorize('deploy', $website);

        $user = Auth::user();
        $deploymentFeatureKey = 'webpilotai_deploy_hosting';
        $deploymentLimitReached = false;
        $maxDeployments = null;
        $currentDeploymentsCount = 0; // Placeholder for actual count

        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $deploymentFeatureKey && isset($feature['limit'])) {
                    $maxDeployments = (int) $feature['limit'];
                    // TODO: Replace this with actual logic to get current deployment count for the period
                    // $currentDeploymentsCount = $user->currentSubscription()->deployments_used_this_period ?? 0;
                    if ($currentDeploymentsCount >= $maxDeployments) {
                        $deploymentLimitReached = true;
                    }
                    break;
                }
            }
        }

        if ($website->status !== Website::STATUS_COMPLETED || !$website->generated_content_path) {
             return redirect()->route('frontend.webpilotai.websites.show', $website->id)
                              ->with('error', 'Website must be successfully generated before deployment.');
        }

        $lastDeploymentDetails = $website->deployment_details;

        return view('webpilotai::user.websites.deploy', compact('website', 'lastDeploymentDetails', 'deploymentLimitReached', 'maxDeployments', 'currentDeploymentsCount'));
    }

    /**
     * Handle the deployment submission.
     */
    public function deploy(Request $request, Website $website): RedirectResponse
    {
        $this->authorize('deploy', $website);
        $user = Auth::user();
        $deploymentFeatureKey = 'webpilotai_deploy_hosting';

        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            $maxDeployments = null;
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $deploymentFeatureKey && isset($feature['limit'])) {
                    $maxDeployments = (int) $feature['limit'];
                    break;
                }
            }
            if (!is_null($maxDeployments)) {
                // TODO: Replace this with actual logic to get current deployment count for the period
                $currentDeploymentsCount = 0; // Placeholder
                if ($currentDeploymentsCount >= $maxDeployments) {
                    return redirect()->route('frontend.webpilotai.websites.deploy.form', $website->id)
                                     ->with('error', "You have reached your deployment limit of {$maxDeployments} for the current period.");
                }
            }
        }

        if ($website->status !== Website::STATUS_COMPLETED || !$website->generated_content_path) {
             return redirect()->route('frontend.webpilotai.websites.show', $website->id)
                              ->with('error', 'Website must be successfully generated before deployment.');
        }

        $validatedData = $request->validate([
            'deployment_type' => 'required|string|in:ftp,sftp,cpanel',
            'host' => 'required|string|max:255',
            'username' => 'required|string|max:255',
            'password' => 'required|string|max:255',
            'port' => 'nullable|integer|min:1|max:65535',
            'remote_path' => 'nullable|string|max:255',
            'cpanel_domain' => 'nullable|string|max:255',
        ]);

        if ($validatedData['deployment_type'] === 'cpanel') {
            $request->validate(['cpanel_domain' => 'required|string|max:255']);
        }

        $deploymentConfig = $validatedData;
        $deploymentConfig['password'] = Crypt::encryptString($validatedData['password']);

        \Modules\WebPilotAI\Jobs\DeployWebsiteToHosting::dispatch($website, $deploymentConfig);

        // TODO: Increment deployment count for the user for the current period.
        return redirect()->route('frontend.webpilotai.websites.show', $website->id)->with('success', 'Deployment process has started.');
    }

    /**
     * Re-queue the website for generation.
     */
    public function regenerate(Website $website): RedirectResponse
    {
        Log::debug("[WebsiteController@regenerate] Attempting to regenerate website ID: {$website->id}. Website User ID: {$website->user_id}. Authenticated User ID: " . Auth::id());

        $this->authorize('update', $website); // User must own the website to regenerate
        $user = Auth::user();
        $websiteRegenerationFeatureKey = 'webpilotai_generate_website'; // Assuming regeneration uses the same feature key and cost

        Log::debug("[WebsiteController@regenerate] Authorization successful for website ID: {$website->id}.");

        // Optional: Check if the website is actually in a failed state
        if ($website->status !== Website::STATUS_FAILED && $website->status !== Website::STATUS_DEPLOYMENT_FAILED) {
            Log::warning("[WebsiteController@regenerate] Website ID: {$website->id} is not in a failed state (current: {$website->status}). Redirecting.");
            return redirect()->route('frontend.webpilotai.websites.index')
                             ->with('error', 'This website is not in a state that allows direct regeneration retry.');
        }

        // --- Credit System Integration for Website Regeneration ---
        $regenerationCost = 0;
        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $cost = $user->currentSubscription()->plan->getFeatureCreditCost($websiteRegenerationFeatureKey);
            if (!is_null($cost)) {
                $regenerationCost = (int) $cost;
            } else {
                Log::warning("Credit cost for feature '{$websiteRegenerationFeatureKey}' (for regeneration) is not defined. Assuming 0 cost for user {$user->id}.");
            }
        } else {
            Log::warning("User {$user->id} does not have an active subscription plan. Credit cost check for '{$websiteRegenerationFeatureKey}' (for regeneration) resulted in 0 cost assumption.");
        }

        $userCreditBalance = $user->current_credit_balance;

        if ($regenerationCost > 0 && $userCreditBalance < $regenerationCost) {
            return redirect()->route('frontend.webpilotai.websites.show', $website->id)
                             ->with('error', "Insufficient credits. You need at least {$regenerationCost} credits to re-generate this website.");
        }
        // Update status and clear previous errors
        $website->update([
            'status' => Website::STATUS_PENDING_REGENERATION,
            'generation_error' => null,
            'deployment_details' => null, // Clear deployment details if it was a deployment failure
        ]);

        // Deduct credits for website regeneration
        if ($regenerationCost > 0) {
            $transactionType = 'spend_feature_webpilotai_regenerate_website'; // Potentially a different type for regeneration
            $description = "AI Website Re-generation: " . ($website->name ?: 'Untitled Website');
            if (!$user->deductCredits($regenerationCost, $transactionType, $description, $website)) {
                Log::critical("Failed to deduct credits for user {$user->id} for website ID {$website->id} during regeneration. Cost: {$regenerationCost}. Regeneration job will still be dispatched.");
            }
        }

        // Dispatch the job to handle AI generation
        ProcessWebsiteGeneration::dispatch($website);

        return redirect()->route('frontend.webpilotai.websites.index')
                         ->with('success', 'Website regeneration has been queued. Please check its status shortly.');
    }

    /**
     * Helper function to format page names into a structured array.
     *
     * @param array|null $pageNames
     * @return array|null
     */
    private function formatPagesStructure(?array $pageNames): ?array
    {
        $filteredPageNames = collect($pageNames)->filter(fn($name) => !empty(trim($name)))->all();

        if (empty($filteredPageNames)) {
            return [
                ['name' => 'Home', 'slug' => 'index', 'filename' => 'index.html'],
                ['name' => 'About Us', 'slug' => 'about', 'filename' => 'about.html'],
                ['name' => 'Contact', 'slug' => 'contact', 'filename' => 'contact.html'],
            ];
        }

        $slugs = [];
        return collect($filteredPageNames)->map(function ($name) use (&$slugs) {
            $trimmedName = trim($name);
            $slug = Str::slug($trimmedName);
            $originalSlug = $slug;
            $counter = 1;

            while (in_array($slug, $slugs)) {
                $slug = $originalSlug . '-' . $counter++;
            }
            $slugs[] = $slug;

            return ['name' => $trimmedName, 'slug' => $slug, 'filename' => $slug . '.html'];
        })->all();
    }
}
