<?php

namespace Modules\WebPilotAI\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\JsonResponse;
use Modules\WebPilotAI\Models\AIModel;
use App\Models\Setting;
use Illuminate\Support\Facades\Validator; // Import Validator
use Illuminate\Support\Facades\Log;

class ContentWriterController extends Controller
{
    /**
     * Display the AI Content Writer page.
     */
    public function index()
    {
        $user = Auth::user();
        $contentWriterFeatureKey = 'webpilotai_ai_content_writer';
        $premiumAccessFeatureKey = 'webpilotai_premium_items_access';

        $canAccessContentWriter = false;
        $canAccessPremiumModels = false;

        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $planFeatures = $user->currentSubscription()->plan->features ?? [];
            foreach ($planFeatures as $feature) {
                if (isset($feature['key']) && $feature['key'] === $contentWriterFeatureKey) {
                    $canAccessContentWriter = true; // Check if the base feature is enabled
                }
                if (isset($feature['key']) && $feature['key'] === $premiumAccessFeatureKey) {
                    $canAccessPremiumModels = true;
                }
            }
        }

        if (!$canAccessContentWriter) {
            // Or redirect with an error message if the feature is entirely disabled for the user's plan
            return redirect()->route('dashboard') // Or a more appropriate route
                             ->with('error', 'AI Content Writer feature is not available on your current plan.');
        }

        $aiModelsQuery = AIModel::where('is_active', true)
            // Assuming text models don't have a specific 'type' yet,
            // or filter by provider if certain providers are text-only
            ->whereIn('provider', ['OpenAI', 'Anthropic', 'GoogleAI']); // Add other text providers

        if (!$canAccessPremiumModels) {
            $aiModelsQuery->where('is_premium', false);
        }
        $aiModels = $aiModelsQuery->orderBy('provider')->orderBy('name')->get();

        $contentTypes = [
            'headline' => 'Headline',
            'paragraph' => 'Paragraph',
            'product_description' => 'Product Description',
            'call_to_action' => 'Call to Action (CTA)',
            'meta_description' => 'SEO Meta Description',
            'email_subject' => 'Email Subject Line',
            'social_media_post' => 'Social Media Post',
        ];

        $tones = [
            'professional' => 'Professional',
            'casual' => 'Casual',
            'witty' => 'Witty',
            'persuasive' => 'Persuasive',
            'informative' => 'Informative',
            'friendly' => 'Friendly',
            'formal' => 'Formal',
        ];

        return view('webpilotai::user.contentwriter.index', compact('aiModels', 'contentTypes', 'tones'));
    }

    /**
     * Generate content using AI.
     */
    public function generate(Request $request): JsonResponse
    {
        $user = Auth::user();
        $contentWriterFeatureKey = 'webpilotai_ai_content_writer';

        // Re-check feature access for the API endpoint
        if (!$user->currentSubscription() || !$user->currentSubscription()->plan->hasFeature($contentWriterFeatureKey)) {
            return response()->json(['error' => 'Access to AI Content Writer denied.'], 403);
        }

        // --- Credit System Integration ---
        $contentGenerationCost = 0; // Default to 0
        if ($user->currentSubscription() && $user->currentSubscription()->plan) {
            $cost = $user->currentSubscription()->plan->getFeatureCreditCost($contentWriterFeatureKey);
            if (!is_null($cost)) {
                $contentGenerationCost = (int) $cost; // Ensure cost is an integer
            } else {
                // If cost is not defined for the feature, log it.
                // Depending on business logic, you might allow free use or return an error.
                Log::warning("Credit cost for feature '{$contentWriterFeatureKey}' is not defined in FeatureCreditCost table. Assuming 0 cost for user {$user->id}.");
            }
        } else {
            // User might not have a subscription, or plan details are missing.
            Log::warning("User {$user->id} does not have an active subscription plan or plan details are missing. Credit cost check for '{$contentWriterFeatureKey}' resulted in 0 cost assumption.");
        }

        // Use the accessor from User model for current credit balance
        $userCreditBalance = $user->current_credit_balance; 

        // Check credits only if the feature has a cost
        if ($contentGenerationCost > 0 && $userCreditBalance < $contentGenerationCost) {
            return response()->json(['error' => "Insufficient credits. You need at least {$contentGenerationCost} credits to generate content."], 402);
        }

        $validator = Validator::make($request->all(), [ // Use Validator facade
            'ai_model_id' => 'required|exists:ai_models,id',
            'topic' => 'required|string|min:10|max:1000',
            'content_type' => 'required|string|max:50',
            'tone' => 'required|string|max:50',
            'keywords' => 'nullable|string|max:255',
            'length_preference' => 'nullable|string|in:short,medium,long',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 422);
        }
        $validated = $validator->validated();

        $aiModel = AIModel::find($validated['ai_model_id']);
        if (!$aiModel || !$aiModel->api_key_setting_name) {
            return response()->json(['error' => 'Selected AI Model is not configured correctly.'], 400);
        }

        $apiKey = Setting::getValue($aiModel->api_key_setting_name);
        if (!$apiKey) {
            return response()->json(['error' => "API key for model '{$aiModel->name}' is not configured."], 400);
        }

        $prompt = "Generate a '{$validated['content_type']}' with a '{$validated['tone']}' tone about the following topic: \"{$validated['topic']}\".";
        if (!empty($validated['keywords'])) {
            $prompt .= " Incorporate these keywords: {$validated['keywords']}.";
        }
        if (!empty($validated['length_preference'])) {
            $prompt .= " The desired length is '{$validated['length_preference']}'.";
        }
        // Add more specific instructions based on content_type if needed

        try {
            // For now, only implementing OpenAI. Expand this for other providers.
            if ($aiModel->provider === 'OpenAI') {
                $client = \OpenAI::client($apiKey);
                $response = $client->chat()->create([
                    'model' => $aiModel->identifier,
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are an expert content writer. Generate concise and relevant content based on the user\'s request.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'temperature' => 0.7,
                    'max_tokens' => ($validated['length_preference'] === 'long' ? 500 : ($validated['length_preference'] === 'medium' ? 250 : 150)),
                ]);
                $generatedText = $response->choices[0]->message->content;

                // Deduct credits only if there was a cost
                if ($contentGenerationCost > 0) {
                    $transactionType = 'spend_feature_webpilotai_ai_content_writer';
                    $description = "AI Content Generation: {$validated['content_type']} - " . \Illuminate\Support\Str::limit($validated['topic'], 50);
                    
                    if (!$user->deductCredits($contentGenerationCost, $transactionType, $description)) {
                        // This case implies an issue with the deduction logic itself or a race condition if balance was sufficient.
                        Log::error("Failed to deduct credits for user {$user->id} after successful content generation. Cost: {$contentGenerationCost}");
                    }
                }

                return response()->json(['success' => true, 'text' => trim($generatedText)]);
            } else {
                return response()->json(['error' => "AI provider '{$aiModel->provider}' not yet supported for content writing."], 400);
            }
        } catch (\Exception $e) {
            Log::error("AI Content Generation Error for user {$user->id} with model {$aiModel->name}: " . $e->getMessage());
            return response()->json(['error' => 'Failed to generate content. Please try again. ' . $e->getMessage()], 500);
        }
    }
}