<?php

namespace Modules\StripeGateway\Providers;

use Illuminate\Support\ServiceProvider;
use Modules\StripeGateway\Services\StripeService; // Import the StripeService
use Modules\StripeGateway\Providers\EventServiceProvider; // Import EventServiceProvider
use Illuminate\Support\Facades\Blade;
use Nwidart\Modules\Traits\PathNamespace;

class StripeGatewayServiceProvider extends ServiceProvider
{
    /**
     * @var string $moduleName
     */
    protected string $moduleName = 'StripeGateway';

    /**
     * @var string $moduleNameLower
     */
    protected string $moduleNameLower = 'stripegateway';

    use PathNamespace; // Add this trait

    public function boot(): void
    {
        $this->registerConfig();
        $this->loadMigrationsFrom(module_path($this->moduleName, 'Database/Migrations'));
        $this->registerTranslations(); // Uncomment and implement
        $this->registerViews();      // Uncomment and implement
        // Route loading is handled by the main app's RouteServiceProvider loop for modules.
        // $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/web.php'));
        // $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/admin.php')); // Note: path should be Routes/admin.php
        // $this->loadRoutesFrom(module_path($this->moduleName, 'Routes/api.php'));
    }
// In Modules\StripeGateway\Providers\StripeGatewayServiceProvider.php
public function register(): void
{
    $this->app->register(RouteServiceProvider::class); // Register module's RouteServiceProvider
    $this->app->register(EventServiceProvider::class); // Register module's EventServiceProvider

    $this->app->singleton(StripeService::class, function ($app) {
        // The StripeService constructor expects keys.
        // We fetch them from settings here.
        $secretKey = function_exists('setting') ? setting('stripe_secret_key') : null;
        $webhookSecret = function_exists('setting') ? setting('stripe_webhook_secret') : null;
        return new StripeService($secretKey, $webhookSecret);
    });
}

    protected function registerConfig(): void
    {
        $this->publishes([
            module_path($this->moduleName, 'config/config.php') => config_path($this->moduleNameLower . '.php'),
        ], 'config');
        $this->mergeConfigFrom(
            module_path($this->moduleName, 'config/config.php'), $this->moduleNameLower
        );
    }

    public function registerViews(): void
    {
        $viewPath = resource_path('views/modules/'.$this->moduleNameLower);
        $sourcePath = module_path($this->moduleName, 'resources/views');

        $this->publishes([$sourcePath => $viewPath], ['views', $this->moduleNameLower.'-module-views']);

        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->moduleNameLower);

        Blade::componentNamespace(config('modules.namespace').'\\' . $this->moduleName . '\\View\\Components', $this->moduleNameLower);
    }

    public function registerTranslations(): void
    {
        $langPath = resource_path('lang/modules/' . $this->moduleNameLower);

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleNameLower);
            $this->loadJsonTranslationsFrom($langPath);
        } else {
            $this->loadTranslationsFrom(module_path($this->moduleName, 'lang'), $this->moduleNameLower);
            $this->loadJsonTranslationsFrom(module_path($this->moduleName, 'lang'));
        }
    }

    public function provides(): array
    {
        return [];
    }

    private function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach (config('view.paths') as $path) {
            if (is_dir($path.'/modules/'.$this->moduleNameLower)) {
                $paths[] = $path.'/modules/'.$this->moduleNameLower;
            }
        }
        return $paths;
    }
}