<?php

namespace Modules\StripeGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\StripeGateway\Services\StripeService;
use App\Services\CreditService;
use Stripe\Event as StripeEvent;
use Stripe\Subscription as StripeSubscriptionObject; // Alias to avoid conflict
use Carbon\Carbon;

class StripeWebhookController extends Controller
{
    protected StripeService $stripeService;
    protected CreditService $creditService;

    public function __construct(StripeService $stripeService, CreditService $creditService)
    {
        $this->stripeService = $stripeService;
        $this->creditService = $creditService;
    }

    public function handleWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $event = null;

        try {
            $event = $this->stripeService->constructWebhookEvent($payload, $sigHeader);
        } catch (\UnexpectedValueException $e) {
            // Invalid payload
            Log::error('Stripe Webhook Error: Invalid payload.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            Log::error('Stripe Webhook Error: Invalid signature.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Invalid signature'], 400);
        } catch (\Exception $e) {
            Log::error('Stripe Webhook Error: Could not verify signature.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Signature verification error'], 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'checkout.session.completed':
                $session = $event->data->object; // contains a \Stripe\Checkout\Session
                Log::info('Stripe Webhook: checkout.session.completed received.', [
                    'session_id' => $session->id,
                    'subscription_id' => $session->subscription,
                    'customer_id' => $session->customer,
                    'payment_status' => $session->payment_status
                ]);
                // Ensure subscription ID exists and payment was successful before processing
                if ($session->subscription && $session->payment_status == 'paid') {
                    $this->handleSubscriptionUpdate($session->subscription, $session->customer);
                } else {
                    Log::warning('Stripe Webhook: checkout.session.completed did not meet criteria for processing.', [
                        'session_id' => $session->id,
                        'has_subscription_id' => !empty($session->subscription),
                        'payment_status' => $session->payment_status
                    ]);
                }
                break;

            case 'customer.subscription.created':
            case 'customer.subscription.updated':
            case 'customer.subscription.resumed':
                $stripeSubscription = $event->data->object; // contains a \Stripe\Subscription
                Log::info('Stripe Webhook: ' . $event->type, ['subscription_id' => $stripeSubscription->id]);
                $this->handleSubscriptionUpdate($stripeSubscription->id, $stripeSubscription->customer);
                break;

            case 'customer.subscription.trial_will_end':
                $stripeSubscription = $event->data->object;
                Log::info('Stripe Webhook: customer.subscription.trial_will_end', ['subscription_id' => $stripeSubscription->id]);
                // TODO: Implement user notification logic here if needed
                break;

            case 'customer.subscription.deleted':
                $stripeSubscription = $event->data->object;
                Log::info('Stripe Webhook: customer.subscription.deleted', ['subscription_id' => $stripeSubscription->id]);
                $this->handleSubscriptionCancellation($stripeSubscription->id, $stripeSubscription);
                break;

            case 'invoice.payment_succeeded':
                $invoice = $event->data->object;
                Log::info('Stripe Webhook: invoice.payment_succeeded', ['invoice_id' => $invoice->id, 'subscription_id' => $invoice->subscription]);
                if ($invoice->subscription) {
                    // This event confirms a recurring payment.
                    // The customer.subscription.updated event usually handles the date changes.
                    // We call handleSubscriptionUpdate to ensure local record is synced and for potential renewal credit awarding.
                    $this->handleSubscriptionUpdate($invoice->subscription, $invoice->customer, true /* isRenewal */);
                }
                break;

            case 'invoice.payment_failed':
                $invoice = $event->data->object;
                Log::error('Stripe Webhook: invoice.payment_failed', ['invoice_id' => $invoice->id, 'subscription_id' => $invoice->subscription]);
                if ($invoice->subscription) {
                    $this->handleSubscriptionPaymentFailure($invoice->subscription);
                }
                break;
            
            default:
                Log::info('Stripe Webhook: Received unhandled event type ' . $event->type, ['event_id' => $event->id]);
        }

        return response()->json(['status' => 'success']);
    }

    protected function handleSubscriptionUpdate($stripeSubscriptionId, $stripeCustomerId, bool $isRenewal = false)
    {
        try {
            if (!setting('stripe_secret_key')) {
                 throw new \Exception('Stripe secret key is not configured for webhook processing.');
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            $stripeSubscription = StripeSubscriptionObject::retrieve($stripeSubscriptionId);

            $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscription->id)->first();

            $user = User::where('stripe_customer_id', $stripeCustomerId)->first();
            if (!$user && $localSubscription) {
                $user = $localSubscription->user;
            }
            
            if (!$user) {
                Log::error("Stripe Webhook (handleSubscriptionUpdate): User not found for Stripe Customer ID: {$stripeCustomerId} or local subscription.", ['stripe_subscription_id' => $stripeSubscriptionId]);
                return;
            }
            
            if (!$user->stripe_customer_id && $stripeCustomerId) {
                $user->stripe_customer_id = $stripeCustomerId;
                $user->save();
                Log::info("Stripe Webhook (handleSubscriptionUpdate): Updated user {$user->id} with stripe_customer_id {$stripeCustomerId}");
            }

            $plan = null;
            if ($localSubscription && $localSubscription->plan) {
                $plan = $localSubscription->plan;
            } elseif (isset($stripeSubscription->metadata->local_plan_id)) { // Corrected metadata key
                $plan = SubscriptionPlan::find($stripeSubscription->metadata->local_plan_id); // Corrected metadata key
            } else {
                // Attempt to get plan from items if metadata is missing (more complex)
                if (isset($stripeSubscription->items->data[0]->price->product)) {
                    $stripeProductId = $stripeSubscription->items->data[0]->price->product;
                    $plan = SubscriptionPlan::where('stripe_product_id', $stripeProductId)->first();
                }
            }

            if ($localSubscription) {
                $oldStatus = $localSubscription->status;
                $localSubscription->status = $stripeSubscription->status;
                $localSubscription->starts_at = Carbon::createFromTimestamp($stripeSubscription->current_period_start);
                $localSubscription->ends_at = Carbon::createFromTimestamp($stripeSubscription->current_period_end);
                $localSubscription->trial_ends_at = $stripeSubscription->trial_end ? Carbon::createFromTimestamp($stripeSubscription->trial_end) : null;
                
                // Handle cancellation details
                if ($stripeSubscription->cancel_at_period_end) {
                    $localSubscription->cancelled_at = $stripeSubscription->canceled_at ? Carbon::createFromTimestamp($stripeSubscription->canceled_at) : now();
                } elseif ($stripeSubscription->status === 'canceled' && !$localSubscription->cancelled_at) {
                    // If Stripe status is 'canceled' but not cancel_at_period_end, it was likely an immediate cancel.
                    $localSubscription->cancelled_at = $stripeSubscription->canceled_at ? Carbon::createFromTimestamp($stripeSubscription->canceled_at) : now();
                }

                $localSubscription->save();
                Log::info("Stripe Webhook: Updated local subscription ID {$localSubscription->id} from status {$oldStatus} to {$stripeSubscription->status}");

                if ($isRenewal && $stripeSubscription->status === 'active' && $plan && function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_renewal > 0) {
                    // Idempotency for credit awarding on renewal needs careful handling.
                    // A common way is to check if credits for this specific billing period have already been awarded.
                    // This might involve storing the last credited period end date or a unique invoice ID.
                    // For simplicity here, we'll assume it's okay or handled by other checks.
                    // Consider adding a check: e.g., if $localSubscription->last_credited_period_end != $stripeSubscription->current_period_end
                    $this->creditService->awardCredits($user, $plan->credits_awarded_on_renewal, 'award_subscription_renewal', "Credits for {$plan->name} renewal", $localSubscription);
                    // $localSubscription->update(['last_credited_period_end' => Carbon::createFromTimestamp($stripeSubscription->current_period_end)]);
                }

            } elseif ($user && $plan) {
                // Idempotency Check: Ensure we don't create duplicate subscriptions
                $existingSubscription = Subscription::where('gateway_subscription_id', $stripeSubscription->id)->first();
                if ($existingSubscription) {
                    Log::info("Stripe Webhook (handleSubscriptionUpdate): Subscription with Stripe ID {$stripeSubscription->id} already exists locally. Skipping creation, will be updated if necessary.");
                    // Optionally, call the update logic again or ensure it's robust enough
                    // $this->handleSubscriptionUpdate($stripeSubscriptionId, $stripeCustomerId); // Be careful of recursion
                    return;
                }

                // Deactivate any other existing active/trialing subscriptions for this user first
                $user->subscriptions()
                    ->where('gateway_subscription_id', '!=', $stripeSubscription->id) // Don't cancel self if somehow re-processed
                    ->whereIn('status', ['active', 'trialing'])
                    ->update([
                        'status' => 'cancelled',
                        'ends_at' => now(),
                        'cancelled_at' => now()
                    ]);

                $newSubscription = Subscription::create([
                    'user_id' => $user->id,
                    'subscription_plan_id' => $plan->id,
                    'payment_gateway' => 'stripegateway',
                    'gateway_subscription_id' => $stripeSubscription->id,
                    'status' => $stripeSubscription->status,
                    'price_at_purchase' => $plan->price,
                    'currency_at_purchase' => $plan->currency,
                    'starts_at' => Carbon::createFromTimestamp($stripeSubscription->current_period_start),
                    'ends_at' => Carbon::createFromTimestamp($stripeSubscription->current_period_end),
                    'trial_ends_at' => $stripeSubscription->trial_end ? Carbon::createFromTimestamp($stripeSubscription->trial_end) : null,
                ]);
                Log::info("Stripe Webhook: Created new local subscription ID {$newSubscription->id} for user {$user->id}, Stripe sub ID {$stripeSubscription->id}");

                if (function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_purchase > 0) {
                    $this->creditService->awardCredits($user, $plan->credits_awarded_on_purchase, 'award_subscription_purchase', "Credits for {$plan->name} subscription", $newSubscription);
                }
                if (!empty($plan->target_role) && class_exists(\Spatie\Permission\Models\Role::class) && \Spatie\Permission\Models\Role::where('name', $plan->target_role)->where('guard_name', 'web')->exists()) {
                    $user->syncRoles([$plan->target_role]);
                }
            } else {
                 Log::warning("Stripe Webhook (handleSubscriptionUpdate): Could not create or update. User or Plan missing.", [
                    'stripe_subscription_id' => $stripeSubscription->id,
                    'user_found' => !is_null($user),
                    'plan_found' => !is_null($plan),
                    'local_subscription_exists' => !is_null($localSubscription)
                 ]);
            }

        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error("Stripe API Error in handleSubscriptionUpdate for sub ID {$stripeSubscriptionId}: " . $e->getMessage(), ['stripe_error_code' => $e->getStripeCode()]);
        } catch (\Exception $e) {
            Log::error("General Error in handleSubscriptionUpdate for sub ID {$stripeSubscriptionId}: " . $e->getMessage(), ["trace" => $e->getTraceAsString()]);
        }
    }

    protected function handleSubscriptionCancellation($stripeSubscriptionId, StripeSubscriptionObject $stripeSubscription = null)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscriptionId)->first();
        if ($localSubscription) {
            $localSubscription->status = 'cancelled';
            $localSubscription->cancelled_at = $stripeSubscription && $stripeSubscription->canceled_at 
                                              ? Carbon::createFromTimestamp($stripeSubscription->canceled_at) 
                                              : now();
            // If Stripe provides a specific end date for a cancellation at period end, use it.
            // The `customer.subscription.updated` event with status `canceled` might be more definitive for `ends_at`.
            // For `customer.subscription.deleted`, if `cancel_at_period_end` was true, `ends_at` should already reflect the period end.
            // If it was an immediate cancellation, `ends_at` should be `canceled_at`.
            if ($stripeSubscription && $stripeSubscription->status === 'canceled') {
                 $localSubscription->ends_at = $localSubscription->cancelled_at;
            }
            // If ends_at is still in the future after this, it means it was set to cancel at period end,
            // and the webhook for 'customer.subscription.updated' (to status 'canceled') should have set the final ends_at.
            // Or, if it's an immediate delete, ends_at becomes now (via cancelled_at).

            $localSubscription->save();
            Log::info("Stripe Webhook: Cancelled local subscription ID {$localSubscription->id}");
        } else {
            Log::warning("Stripe Webhook: Received cancellation for unknown Stripe subscription ID: {$stripeSubscriptionId}");
        }
    }

    protected function handleSubscriptionPaymentFailure($stripeSubscriptionId)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscriptionId)->first();
        if ($localSubscription) {
            if (!setting('stripe_secret_key')) {
                 Log::error('Stripe Webhook (handleSubscriptionPaymentFailure): Stripe secret key not set.');
                 return;
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            try {
                $stripeSub = StripeSubscriptionObject::retrieve($stripeSubscriptionId);
                $localSubscription->status = $stripeSub->status; // e.g., 'past_due', 'unpaid'
                $localSubscription->save();
                Log::info("Stripe Webhook: Updated local subscription ID {$localSubscription->id} to status {$stripeSub->status} due to payment failure.");
                // TODO: Implement user notification about payment failure
            } catch (\Exception $e) {
                Log::error("Stripe Webhook (handleSubscriptionPaymentFailure): Error fetching Stripe subscription {$stripeSubscriptionId}: " . $e->getMessage());
            }
        } else {
            Log::warning("Stripe Webhook: Received payment failure for unknown Stripe subscription ID: {$stripeSubscriptionId}");
        }
    }
}
