<?php

namespace Modules\PaystackGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\PaystackGateway\Services\PaystackService;
// Potentially add: use App\Services\CreditService;
// Potentially add: use App\Services\WalletService;

class PaystackWebhookController extends Controller
{
    protected PaystackService $paystackService;

    public function __construct(PaystackService $paystackService)
    {
        $this->paystackService = $paystackService;
    }

    public function handleWebhook(Request $request)
    {
        // 1. Verify the webhook signature
        $signature = $request->header('x-paystack-signature');
        $rawBody = $request->getContent();

        if (!$this->paystackService->secretKey || !$signature) {
            Log::warning('Paystack Webhook: Missing secret key or signature for verification.');
            return response()->json(['status' => 'error', 'message' => 'Configuration error or missing signature'], 400);
        }

        $calculatedSignature = hash_hmac('sha512', $rawBody, $this->paystackService->secretKey);

        if (!hash_equals($calculatedSignature, $signature)) {
            Log::warning('Paystack Webhook: Invalid signature.', [
                'received_signature' => $signature,
                'calculated_signature' => $calculatedSignature,
                'ip_address' => $request->ip(),
            ]);
            return response()->json(['status' => 'error', 'message' => 'Signature verification failed'], 401);
        }

        // 2. Process the event
        $event = json_decode($rawBody);

        if (!$event || !isset($event->event)) {
            Log::error('Paystack Webhook: Invalid event data received or missing event type.');
            return response()->json(['status' => 'error', 'message' => 'Invalid event data'], 400);
        }

        Log::info('Paystack Webhook Received:', ['event_type' => $event->event, 'data' => (array)($event->data ?? [])]);

        $eventType = $event->event;
        $data = $event->data ?? null;

        switch ($eventType) {
            case 'charge.success':
                // This is a crucial event for recurring payments or successful one-time charges.
                if ($data && isset($data->reference)) {
                    // If it's related to a subscription, the metadata might contain subscription_id or plan_code.
                    // You might need to fetch the local subscription using $data->reference or metadata.
                    // For now, let's assume it might be a subscription payment.
                    $this->handleSuccessfulCharge($data);
                }
                break;

            case 'subscription.create': // A new subscription is created on Paystack
            case 'subscription.enable': // A previously disabled subscription is re-enabled
                // You might want to update your local subscription status if it was, for example, 'pending' or 'disabled'.
                // $data->subscription_code, $data->customer->email, $data->plan->plan_code
                break;

            case 'subscription.disable': // Subscription is disabled (e.g., due to payment failure or manual action)
                // Update your local subscription status to 'cancelled' or 'suspended'.
                // $data->subscription_code
                break;

            // Add more Paystack event types as needed:
            // - invoice.create
            // - invoice.update
            // - invoice.payment_failed
            // - transfer.success, transfer.failed (if using Paystack transfers)

            default:
                Log::info('Paystack Webhook: Received unhandled event type: ' . $eventType);
        }

        return response()->json(['status' => 'success'], 200);
    }

    protected function handleSuccessfulCharge($data)
    {
        // Example: Find subscription by reference (if you store Paystack's transaction reference)
        // $subscription = Subscription::where('gateway_transaction_id', $data->reference)->first();
        // if ($subscription && $subscription->status !== 'active') {
        //     $subscription->status = 'active';
        //     // Potentially update ends_at based on the new payment
        //     $subscription->save();
        //     Log::info("Paystack Webhook: Subscription ID {$subscription->id} marked active via charge.success for reference {$data->reference}.");
        // }
        Log::info("Paystack Webhook: Successful charge processed for reference: {$data->reference}", (array)$data);
        // If this charge is for a subscription renewal, you'd update the subscription's `ends_at` date.
    }
}