<?php

namespace Modules\PaypalGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\PaypalGateway\Services\PaypalService;

class PaypalWebhookController extends Controller
{
    protected PaypalService $paypalService;

    public function __construct(PaypalService $paypalService)
    {
        $this->paypalService = $paypalService;
    }

    public function handleWebhook(Request $request)
    {
        $rawBody = $request->getContent();
        $headers = $request->headers->all(); // Get all headers as an array

        // **IMPORTANT: Implement robust webhook signature verification here!**
        // The $this->paypalService->verifyWebhookSignature() is a placeholder.
        // You MUST verify the signature according to PayPal's documentation.
        if (!$this->paypalService->verifyWebhookSignature($headers, $rawBody)) {
            Log::warning('PayPal Webhook: Invalid signature or verification failed. IP: ' . $request->ip());
            return response()->json(['status' => 'error', 'message' => 'Signature verification failed'], 401); // Use 401 Unauthorized
        }

        $event = json_decode($rawBody);

        if (!$event || !isset($event->event_type)) {
            Log::error('PayPal Webhook: Invalid event data received.');
            return response()->json(['status' => 'error', 'message' => 'Invalid event data'], 400);
        }

        Log::info('PayPal Webhook Received:', (array) $event);

        $eventType = strtoupper($event->event_type); // PayPal event types are usually uppercase
        $resource = $event->resource ?? null;

        switch ($eventType) {
            case 'BILLING.SUBSCRIPTION.ACTIVATED':
            case 'BILLING.SUBSCRIPTION.UPDATED':
                if ($resource && isset($resource->id)) {
                    $this->handleSubscriptionUpdate($resource);
                }
                break;

            case 'BILLING.SUBSCRIPTION.CANCELLED':
            case 'BILLING.SUBSCRIPTION.EXPIRED':
            case 'BILLING.SUBSCRIPTION.SUSPENDED':
                if ($resource && isset($resource->id)) {
                    $this->handleSubscriptionStateChange($resource, strtolower($resource->status ?? 'cancelled'));
                }
                break;

            case 'PAYMENT.SALE.COMPLETED': // For recurring payments
                if ($resource && isset($resource->billing_agreement_id)) {
                    // This event confirms a recurring payment.
                    // The BILLING.SUBSCRIPTION.UPDATED event usually handles date changes.
                    // You might want to log this or ensure the subscription is still active.
                    $this->ensureSubscriptionActive($resource->billing_agreement_id);
                }
                break;

            // Add more event types as needed based on PayPal's documentation
            // e.g., BILLING.SUBSCRIPTION.PAYMENT.FAILED

            default:
                Log::info('PayPal Webhook: Received unhandled event type ' . $eventType);
        }

        return response()->json(['status' => 'success']);
    }

    protected function handleSubscriptionUpdate($paypalSubscriptionResource)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $paypalSubscriptionResource->id)->first();
        if ($localSubscription) {
            $localSubscription->status = strtolower($paypalSubscriptionResource->status); // e.g., 'active'
            // Update start_time, agreement_details.next_billing_date etc.
            // $localSubscription->starts_at = \Carbon\Carbon::parse($paypalSubscriptionResource->start_time);
            // $localSubscription->ends_at = isset($paypalSubscriptionResource->agreement_details->next_billing_date) ? \Carbon\Carbon::parse($paypalSubscriptionResource->agreement_details->next_billing_date) : null;
            $localSubscription->save();
            Log::info("PayPal Webhook: Updated local subscription ID {$localSubscription->id} to status {$localSubscription->status}");
        } else {
            // Potentially create a new subscription if it's an activation and doesn't exist
            Log::warning("PayPal Webhook: Received update for unknown PayPal subscription ID: {$paypalSubscriptionResource->id}");
        }
    }

    protected function handleSubscriptionStateChange($paypalSubscriptionResource, string $newStatus)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $paypalSubscriptionResource->id)->first();
        if ($localSubscription) {
            $localSubscription->status = $newStatus;
            if (in_array($newStatus, ['cancelled', 'expired', 'suspended'])) {
                $localSubscription->cancelled_at = $localSubscription->cancelled_at ?? now();
            }
            $localSubscription->save();
            Log::info("PayPal Webhook: Changed local subscription ID {$localSubscription->id} to status {$newStatus}");
        }
    }

    protected function ensureSubscriptionActive(string $paypalSubscriptionId) {
        $localSubscription = Subscription::where('gateway_subscription_id', $paypalSubscriptionId)->first();
        if ($localSubscription && $localSubscription->status !== 'active') {
            // Potentially reactivate or log if a payment comes through for an inactive sub
            Log::info("PayPal Webhook: Payment received for subscription {$paypalSubscriptionId}, current local status: {$localSubscription->status}");
        }
    }
}