<?php

namespace Modules\PaypalGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Modules\PaypalGateway\Services\PaypalService;
use App\Services\CreditService; // Added
use App\Services\WalletService; // Added

class PaypalSubscriptionController extends Controller
{
    protected PaypalService $paypalService;
    protected CreditService $creditService; // Added
    protected WalletService $walletService; // Added

    public function __construct(PaypalService $paypalService, CreditService $creditService, WalletService $walletService)
    {
        $this->paypalService = $paypalService;
        $this->creditService = $creditService; // Added
        $this->walletService = $walletService; // Added
    }

    public function executeSubscription(Request $request)
    {
        $paypalSubscriptionId = $request->query('subscription_id'); // PayPal returns subscription_id on approval
        // The 'ba_token' and 'token' might also be present for older flows or different contexts.
        // For modern Subscriptions API, subscription_id is key.

        Log::info("PayPal ExecuteSubscription: Reached callback.", ['query_params' => $request->query(), 'user_id' => Auth::id(), 'session_dump' => session()->all()]);

        if (!$paypalSubscriptionId) {
            Log::error("PayPal ExecuteSubscription: 'subscription_id' is missing from PayPal callback.", ['query_params' => $request->query()]);
            return redirect()->route('subscription.plans')->with('error', 'Invalid PayPal subscription approval.');
        }
        try {
            $paypalSubscriptionDetails = $this->paypalService->getPaypalSubscriptionDetails($paypalSubscriptionId);

            if ($paypalSubscriptionDetails && in_array(strtoupper($paypalSubscriptionDetails->status), ['ACTIVE', 'APPROVAL_PENDING'])) {
                /** @var User $user */
                // User might not be available directly from Auth if it's a webhook context later,
                // but for direct redirect, Auth::user() is fine.
                // For webhooks, you'd typically get user_id from metadata or the local subscription.
                $user = Auth::user();

                if (!$user) {
                    Log::error("PayPal ExecuteSubscription: User not authenticated on callback.", ['paypal_subscription_id' => $paypalSubscriptionId]);
                    return redirect()->route('login')->with('error', 'Your session expired. Please log in and try subscribing again.');
                }

                // Find the pending local subscription
                Log::info("PayPal ExecuteSubscription: Attempting to find local pending subscription.", ['gateway_subscription_id' => $paypalSubscriptionId, 'user_id' => $user->id, 'status_expected' => 'pending_approval']);
                $localSubscription = Subscription::where('gateway_subscription_id', $paypalSubscriptionId)
                                        ->where('status', 'pending_approval')
                                        ->where('user_id', $user->id) // Ensure it's for the current user
                                        ->first();

                if ($localSubscription) {
                    Log::info("PayPal ExecuteSubscription: Found local pending subscription.", ['local_subscription_id' => $localSubscription->id]);
                    // Deactivate other existing subscriptions for this user
                    $user->subscriptions()
                        ->where('id', '!=', $localSubscription->id)
                        ->whereIn('status', ['active', 'trialing', 'ACTIVE'])
                        ->update(['status' => 'cancelled', 'ends_at' => now(), 'cancelled_at' => now()]);

                    $localSubscription->status = strtolower($paypalSubscriptionDetails->status); // 'active'
                    $localSubscription->starts_at = isset($paypalSubscriptionDetails->start_time) ? \Carbon\Carbon::parse($paypalSubscriptionDetails->start_time) : now();
                    
                    $plan = $localSubscription->plan;
                    if ($plan) {
                        if ($plan->trial_period_days > 0 && strtoupper($paypalSubscriptionDetails->status) === 'ACTIVE') { // Or check if trial info is in $paypalSubscriptionDetails
                            $localSubscription->trial_ends_at = $localSubscription->starts_at->copy()->addDays($plan->trial_period_days);
                            // Determine ends_at based on trial and plan interval
                            $billingCycleAnchor = $localSubscription->trial_ends_at;
                        } else {
                            $billingCycleAnchor = $localSubscription->starts_at;
                        }
                        $localSubscription->ends_at = $billingCycleAnchor->copy()->add($plan->interval, $plan->interval_count);

                        // Award credits
                        if (function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_purchase > 0) {
                            $this->creditService->awardCredits($user, $plan->credits_awarded_on_purchase, 'award_subscription_purchase', "Credits for {$plan->name} subscription", $localSubscription);
                        }
                        // Assign target role
                        if (!empty($plan->target_role) && class_exists(\Spatie\Permission\Models\Role::class) && \Spatie\Permission\Models\Role::where('name', $plan->target_role)->where('guard_name', 'web')->exists()) {
                            $user->syncRoles([$plan->target_role]);
                        }
                    }

                    $localSubscription->save();

                    return redirect()->route('dashboard')->with('success', 'PayPal subscription successfully activated!');
                } else {
                    Log::error("PayPal ExecuteSubscription: Local pending subscription NOT FOUND for PayPal ID: {$paypalSubscriptionId}", ['user_id' => $user->id, 'paypal_subscription_id' => $paypalSubscriptionId]);
                    return redirect()->route('subscription.plans')->with('error', 'Could not find your pending PayPal subscription record.');
                }
            } else {
                Log::error("PayPal ExecuteSubscription: PayPal subscription status not ACTIVE/APPROVAL_PENDING or details not found.", ['paypal_subscription_id' => $paypalSubscriptionId, 'details_status' => $paypalSubscriptionDetails->status ?? 'N/A', 'raw_details' => (array)$paypalSubscriptionDetails]);
                return redirect()->route('subscription.plans')->with('error', 'PayPal subscription could not be activated.');
            }
        } catch (\Exception $e) {
            Log::error("PayPal ExecuteSubscription: General Exception: " . $e->getMessage(), ['paypal_subscription_id' => $paypalSubscriptionId, 'trace' => $e->getTraceAsString()]);
            return redirect()->route('subscription.plans')->with('error', 'There was an issue activating your PayPal subscription: ' . $e->getMessage());
        }
    }

    public function cancelSubscriptionPage(Request $request)
    {
        // User cancelled from PayPal's approval page
        return redirect()->route('subscription.plans')->with('info', 'Your PayPal subscription process was cancelled.');
    }

    public function cancelActiveSubscription(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $activeSubscription = $user->subscriptions()
            ->where('payment_gateway', 'paypalgateway') // Consistent key
            ->where('status', 'active') // Or other active-like statuses
            ->orderBy('created_at', 'desc')
            ->first();

        if (!$activeSubscription) {
            return redirect()->back()->with('info', 'No active PayPal subscription found to cancel.');
        }

        if ($this->paypalService->cancelSubscriptionAtGateway($activeSubscription)) {
            $activeSubscription->status = 'cancelled';
            $activeSubscription->cancelled_at = now();
            $activeSubscription->save();
            return redirect()->back()->with('success', 'Your PayPal subscription has been cancelled.');
        }
        return redirect()->back()->with('error', 'Failed to cancel PayPal subscription. Please try again or contact support.');
    }

    // --- Wallet Deposit Methods ---

    public function initializeWalletDeposit(Request $request)
    {
        if (setting('paypal_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'PayPal deposits are currently disabled.');
        }

        $request->validate(['amount' => 'required|numeric|min:1']);
        $amount = (float) $request->input('amount');
        /** @var User $user */
        $user = Auth::user();
        $currency = strtoupper(setting('currency_code', 'USD'));

        try {
            $response = $this->paypalService->createWalletDepositOrder($user, $amount, $currency);

            if ($response && $response->statusCode == 201) { // 201 Created
                $order = $response->result;
                // Store PayPal Order ID in session to verify on callback
                session(['paypal_wallet_deposit_order_id' => $order->id]);

                foreach ($order->links as $link) {
                    if ($link->rel == 'approve') {
                        return redirect()->away($link->href);
                    }
                }
            }
            Log::error('PayPal Initialize Wallet Deposit Failed', ['response_status' => $response->statusCode ?? 'N/A', 'response_body' => $response->result ?? 'N/A']);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Could not initiate PayPal deposit. Please try again.');
        } catch (\Exception $e) {
            Log::error("PayPal Initialize Wallet Deposit Error for user {$user->id}, amount {$amount}: " . $e->getMessage());
            return redirect()->route('user.wallet.deposit.form')->with('error', 'An error occurred while initiating deposit with PayPal: ' . $e->getMessage());
        }
    }

    public function handleWalletDepositCallback(Request $request)
    {
        $paypalOrderId = session('paypal_wallet_deposit_order_id');
        $token = $request->query('token'); // This is the PayPal Order ID
        $payerId = $request->query('PayerID'); // PayerID is also sent

        /** @var User $user */
        $user = Auth::user();

        if (!$paypalOrderId || $paypalOrderId !== $token) {
            Log::error("PayPal Wallet Deposit Callback: Session Order ID mismatch or missing.", ['session_order_id' => $paypalOrderId, 'token' => $token]);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Invalid PayPal session for deposit.');
        }

        try {
            $response = $this->paypalService->captureWalletDepositOrder($paypalOrderId);

            if ($response && $response->statusCode == 201 && isset($response->result->status) && $response->result->status == 'COMPLETED') {
                $capture = $response->result->purchase_units[0]->payments->captures[0];
                $amountDeposited = (float) $capture->amount->value;
                $currency = $capture->amount->currency_code;

                // Idempotency check
                $existingTransaction = \App\Models\WalletTransaction::where('gateway_transaction_id', $capture->id)->first();
                if ($existingTransaction) {
                    Log::info("PayPal Wallet Deposit: Capture ID {$capture->id} already processed.");
                    return redirect()->route('user.wallet.history')->with('success', 'Your deposit was successful.');
                }

                $this->walletService->deposit($user, $amountDeposited, $currency, 'paypalgateway', $capture->id, "Wallet deposit via PayPal");
                session()->forget('paypal_wallet_deposit_order_id');
                return redirect()->route('user.wallet.history')->with('success', 'Successfully deposited ' . $currency . ' ' . number_format($amountDeposited, 2) . ' to your wallet.');
            }
            Log::error('PayPal Wallet Deposit Capture Failed', ['response_status' => $response->statusCode ?? 'N/A', 'response_body' => $response->result ?? 'N/A']);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'PayPal payment capture failed for deposit.');
        } catch (\Exception $e) {
            Log::error("PayPal Wallet Deposit Callback Error: " . $e->getMessage(), ['paypal_order_id' => $paypalOrderId]);
            return redirect()->route('user.wallet.deposit.form')->with('error', 'An error occurred during PayPal deposit verification: ' . $e->getMessage());
        }
    }

    public function handleWalletDepositCancel(Request $request)
    {
        session()->forget('paypal_wallet_deposit_order_id');
        return redirect()->route('user.wallet.deposit.form')->with('info', 'Your PayPal wallet deposit was cancelled.');
    }
}