# LandingPage Module

## Overview

The LandingPage module is the heart of your website's frontend presentation. It provides a robust system for creating, managing, and displaying dynamic pages with a flexible theming engine. This module allows administrators to control the site's appearance by choosing from available themes and to manage page content, SEO metadata, and page structure. It's designed to be SEO-friendly and can integrate with other modules to showcase diverse content.

## Key Features

-   **Dynamic Page Management:**
    -   Full CRUD (Create, Read, Update, Delete) functionality for web pages via the admin panel.
    -   Each page can have a unique title, slug (URL), and main content (HTML, Markdown, or structured via a WYSIWYG editor).
    -   SEO settings per page: meta title, meta description, meta keywords.
    -   Ability to publish/unpublish pages and set a sort order for navigation or listings.
    -   Option to assign specific theme templates to individual pages for varied layouts.
-   **Powerful Theming System:**
    -   Themes are self-contained units located in `Modules/LandingPage/resources/views/themes/`.
    -   Administrators can easily switch the active theme for the entire website via the admin panel.
    -   Each theme can have its own layouts, partials (headers, footers), page templates, and theme-specific assets (CSS, JS, images).
    -   Themes are designed to be easily created and customized.
-   **SEO & Schema Markup:**
    -   Built-in support for essential SEO meta tags on a per-page basis.
    -   Global SEO settings (site name, title suffix, global meta description/keywords) can be configured.
    -   Theme layouts include placeholders for basic Schema.org markup (JSON-LD), which can be extended for specific page types (e.g., Article, Service, Product).
-   **Content Aggregation:**
    -   Themes are designed to fetch and display content from other installed and enabled modules. For example, showcasing projects from the `DigitalVocano` module or job listings from a future `Jobs` module.
-   **Responsive Design:**
    -   While responsiveness is primarily handled by the CSS within each theme (and the main application's Tailwind CSS setup), the module structure supports creating fully responsive themes.
-   **Blade Components:**
    -   Encourages the use of reusable Blade components within themes for common UI elements like cards, buttons, modals, and hero sections, promoting cleaner and more maintainable theme code.
-   **Theme-Specific Assets & Options (Advanced):**
    -   Supports publishing and linking theme-specific CSS and JavaScript files.
    -   The architecture allows for future extension to include theme-specific configuration options manageable through the admin panel.

## Directory Structure (Key Components)

-   `Http/Controllers/Admin/LpPageController.php`: Manages CRUD operations for dynamic pages.
-   `Http/Controllers/Admin/ThemeSettingController.php`: Handles admin interface for selecting the active theme and managing theme-specific options (if implemented).
-   `Http/Controllers/FrontendPageController.php`: Responsible for fetching the correct `LpPage` entity and rendering it using the currently active theme and appropriate page template.
-   `Entities/LpPage.php`: The Eloquent model representing a dynamic page with its attributes (title, slug, content, meta fields, etc.).
-   `Database/Migrations/`: Contains the migration for the `lp_pages` table.
-   `Database/Seeders/`:
    -   `LpPageSeeder.php`: Seeds default pages (e.g., home, about-us, contact, privacy-policy).
    -   `LandingPageDefaultSettingsSeeder.php`: Seeds essential global settings related to the landing page module (e.g., active theme, homepage slug, global SEO defaults).
    -   `LandingPageDatabaseSeeder.php`: The main seeder for the module, calling other specific seeders.
-   `Routes/admin.php`: Defines routes for the admin panel sections related to page management and theme settings.
-   `Routes/web.php`: Contains the frontend route (typically a catch-all like `/{slug?}`) that directs requests to the `FrontendPageController`.
-   `resources/views/admin/`: Blade views for the admin interface (managing pages, theme settings).
-   `resources/views/themes/`: The parent directory for all available themes.
    -   `themes/[theme_name]/layouts/app.blade.php`: Main layout for a specific theme.
    -   `themes/[theme_name]/partials/`: Reusable parts like headers, footers.
    -   `themes/[theme_name]/pages/` or `templates/`: Specific page templates (e.g., `home.blade.php`, `page.blade.php`, `solution_detail.blade.php`).
-   `resources/views/components/themes/[theme_name]/`: (Recommended) Location for theme-specific anonymous Blade components.
-   `public/`: (Optional, within the module) For theme-specific assets (CSS, JS, images) that need to be published to the main application's `public` directory.

## Setup & Configuration

1.  **Module Installation:**
    The module is typically created using the Artisan command:
    ```bash
    php artisan module:make LandingPage
    ```

2.  **Database Migrations:**
    Run the module's migrations to create the `lp_pages` table and any other necessary schema changes:
    ```bash
    php artisan module:migrate LandingPage
    ```
    Also, ensure any related migrations in the main application (e.g., for adding the `landingpage_active_theme` setting to the `settings` table) are run:
    ```bash
    php artisan migrate
    ```

3.  **Database Seeding:**
    To populate default pages and settings:
    -   Ensure your main `database/seeders/DatabaseSeeder.php` calls the module's seeder:
        ```php
        // In database/seeders/DatabaseSeeder.php
        if (class_exists(\Modules\LandingPage\Database\Seeders\LandingPageDatabaseSeeder::class)) {
            $this->call(\Modules\LandingPage\Database\Seeders\LandingPageDatabaseSeeder::class);
        }
        ```
    -   Then run the main seeder:
        ```bash
        php artisan db:seed
        ```
    -   Alternatively, to seed only this module:
        ```bash
        php artisan module:seed LandingPage
        ```

4.  **Admin Configuration:**
    -   **Manage Pages:** Access the admin panel (typically `yourdomain.com/admin`) and navigate to the "Manage Pages" section (route defined in `module.json`, e.g., `admin.landingpage.pages.index`). Here you can create, edit, and delete pages.
    -   **Theme Settings:** Navigate to "Theme Settings" (route defined in `module.json`, e.g., `admin.landingpage.theme.settings.edit`) to select the active theme for your website.
    -   **Global Site Settings:** Configure general site settings like `site_name`, `site_logo`, `site_favicon`, `site_tagline`, global SEO meta tags, and contact information through your main application's settings management interface. These are often used by the theme's header and footer.

5.  **Asset Publishing (for Theme-Specific Assets):**
    -   If a theme includes its own CSS, JavaScript, or image files within its `Modules/LandingPage/public/themes/[theme_name]/` directory, these assets need to be published to the main application's `public` directory.
    -   The `LandingPageServiceProvider` includes a publish command for this. Run:
        ```bash
        php artisan vendor:publish --tag=landingpage-module-assets --force
        ```
    -   Ensure these published assets are correctly linked in the theme's layout files (e.g., `themes/[theme_name]/layouts/app.blade.php`).

6.  **Route Order (Important):**
    -   The LandingPage module uses a catch-all web route (e.g., `Route::get('/{slug?}', ...)`). To ensure it doesn't override other specific routes from your application or other modules, this route should be registered **late** in your application's route processing order.
    -   This might involve adjusting the order in which module service providers are registered or the order of route file loading in your main `app/Providers/RouteServiceProvider.php` or the module's `RouteServiceProvider`. Setting a higher `priority` in the `LandingPage/module.json` (e.g., `1000`) can also help ensure its routes are loaded later if your module loading respects priority.

7.  **Clear Cache:**
    After making configuration changes, creating new files, or publishing assets, always clear your application's cache:
    ```bash
    php artisan optimize:clear
    ```

## Creating a New Theme

1.  **Create Directory:** Inside `Modules/LandingPage/resources/views/themes/`, create a new directory for your theme (e.g., `my_custom_theme`).
2.  **Core Files:** Within your new theme directory, create the essential Blade files:
    -   `layouts/app.blade.php` (the main layout for this theme)
    -   `partials/header.blade.php`
    -   `partials/footer.blade.php`
    -   `pages/home.blade.php` (or `home.blade.php` directly in the theme root) for the homepage.
    -   `pages/page.blade.php` (or `page.blade.php` directly in the theme root) as the default template for dynamic pages.
    -   (Optional) `templates/` subdirectory for alternative page layouts (e.g., `full_width.blade.php`, `sidebar_left.blade.php`).
3.  **Theme Logic:** Design your HTML structure, styling (using Tailwind CSS from the main app or theme-specific CSS), and any JavaScript interactions.
4.  **Activate Theme:** The new theme directory will automatically be detected and listed in the "Theme Settings" dropdown in the admin panel. Select and save it to make it active.
5.  **View Naming:** When including partials or extending layouts within your theme, use the `$activeTheme` variable passed to views:
    ```blade
    @extends("landingpage::themes.{$activeTheme}.layouts.app")
    @include("landingpage::themes.{$activeTheme}.partials.header")
    ```

## Content Aggregation from Other Modules

Themes can display content from other modules in several ways:

-   **Direct Model Queries:** In your theme's Blade files (especially for sections on `home.blade.php` or specific pages), you can directly query Eloquent models from other enabled modules.
    ```php
    // Example in a Blade file
    @php
        $projects = \Modules\DigitalVocano\Entities\Project::where('is_published', true)->take(3)->get();
    @endphp
    @foreach($projects as $project)
        {{-- Display project --}}
    @endforeach
    ```
-   **Service Classes:** For more complex data fetching or business logic, create service classes within the respective modules. These services can then be injected into the `FrontendPageController` or used via Laravel's service container in Blade (less common for direct Blade usage).
-   **View Composers:** Use Laravel's View Composers to automatically bind data from other modules to specific theme views or partials. This helps keep your Blade templates cleaner.

## SEO & Schema Markup

-   The main theme layout (`layouts/app.blade.php`) should dynamically set `<title>`, `<meta name="description">`, and `<meta name="keywords">` using `@yield` directives.
-   The `FrontendPageController` passes the `$page` object (an `LpPage` instance) to the view. Your page templates (e.g., `page.blade.php`, `home.blade.php`) should then use `@section` to provide specific meta information from the `$page` object, falling back to global settings if page-specific meta is not set.
-   Use the `@push('schema_markup')` stack in your theme layout to allow individual page templates to add specific JSON-LD schema.org markup (e.g., for `Article`, `Product`, `Service`, `Event`).

## Dependencies

-   Primarily relies on the main application's Laravel framework and its configured database.
-   Assumes Tailwind CSS (or another CSS framework) is set up globally via Vite in the main application, and themes will leverage these global styles. Themes can add their own specific styles on top.
-   May interact with other custom modules (like `DigitalVocano`) for content display.

This README provides a solid overview for developers working with or extending the LandingPage module.
```