<?php

namespace Modules\LandingPage\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Modules\LandingPage\Entities\LpPage;
use Modules\LandingPage\Http\Requests\StoreLpPageApiRequest; // You'll need to create this
use Modules\LandingPage\Http\Requests\UpdateLpPageApiRequest; // You'll need to create this
use Modules\LandingPage\Transformers\LpPageResource; // You'll need to create this

class LandingPageController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        // Authorize: Can the current user (or guest if allowed by policy) view any LpPages?
        $this->authorize('viewAny', LpPage::class);

        $query = LpPage::query();

        if ($request->has('is_published')) {
            $query->where('is_published', (bool)$request->input('is_published'));
        }

        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('slug', 'like', "%{$searchTerm}%");
            });
        }

        $pages = $query->orderBy('sort_order')->paginate($request->input('per_page', 15));

        return LpPageResource::collection($pages)->response();
    }

    /**
     * Store a newly created resource in storage.
     * @param StoreLpPageApiRequest $request
     * @return JsonResponse
     */
    public function store(StoreLpPageApiRequest $request): JsonResponse
    {
        // Authorize: Can the current user create an LpPage?
        $this->authorize('create', LpPage::class);

        $validatedData = $request->validated();
        $validatedData['slug'] = $validatedData['slug'] ?? Str::slug($validatedData['title']);

        // Ensure slug uniqueness
        $originalSlug = $validatedData['slug'];
        $count = 1;
        while (LpPage::where('slug', $validatedData['slug'])->exists()) {
            $validatedData['slug'] = $originalSlug . '-' . $count++;
        }

        $page = LpPage::create($validatedData);

        return (new LpPageResource($page))
                ->response()
                ->setStatusCode(201);
    }

    /**
     * Display the specified resource.
     * @param LpPage $landingpage // Route model binding by 'landingpage' (from apiResource name)
     * @return JsonResponse
     */
    public function show(LpPage $landingpage): JsonResponse // Parameter name matches route segment
    {
        // Authorize: Can the current user (or guest) view this specific LpPage?
        $this->authorize('view', $landingpage);
        return (new LpPageResource($landingpage))->response();
    }

    /**
     * Update the specified resource in storage.
     * @param UpdateLpPageApiRequest $request
     * @param LpPage $landingpage
     * @return JsonResponse
     */
    public function update(UpdateLpPageApiRequest $request, LpPage $landingpage): JsonResponse
    {
        // Authorize: Can the current user update this LpPage?
        $this->authorize('update', $landingpage);

        $validatedData = $request->validated();
        if (isset($validatedData['title']) && !isset($validatedData['slug'])) {
            $validatedData['slug'] = Str::slug($validatedData['title']);
        }

        // Ensure slug uniqueness if changed
        if (isset($validatedData['slug']) && $landingpage->slug !== $validatedData['slug']) {
            $originalSlug = $validatedData['slug'];
            $count = 1;
            while (LpPage::where('slug', $validatedData['slug'])->where('id', '!=', $landingpage->id)->exists()) {
                $validatedData['slug'] = $originalSlug . '-' . $count++;
            }
        }

        $landingpage->update($validatedData);

        return (new LpPageResource($landingpage))->response();
    }

    /**
     * Remove the specified resource from storage.
     * @param LpPage $landingpage
     * @return JsonResponse
     */
    public function destroy(LpPage $landingpage): JsonResponse
    {
        // Authorize: Can the current user delete this LpPage?
        $this->authorize('delete', $landingpage);

        $landingpage->delete();

        return response()->json(null, 204);
    }
}