<?php

namespace Modules\LandingPage\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\LandingPage\Entities\Menu;
use Modules\LandingPage\Entities\LpPage; // Add this
use Illuminate\Support\Str;

class MenuController extends Controller
{
    /**
     * Get the defined menu locations.
     *
     * @return array
     */
    protected function getMenuLocations(): array
    {
        return config('landingpage.menu_locations', []);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function index()
    {
        $menus = Menu::orderBy('name')->paginate(15);
        return view('landingpage::admin.menus.index', compact('menus'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function create()
    {
        $menu = new Menu();
        $menuLocations = $this->getMenuLocations();
        return view('landingpage::admin.menus.create', compact('menu', 'menuLocations'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $menuLocationsKeys = array_keys($this->getMenuLocations());
        $request->validate([
            'name' => 'required|string|max:255|unique:lp_menus,name',
            'location' => 'nullable|string|max:255|unique:lp_menus,location|in:' . implode(',', $menuLocationsKeys),
        ]);

        Menu::create([
            'name' => $request->input('name'),
            // Slug will be auto-generated by the Sluggable trait in the Menu model
            'location' => $request->input('location'),
        ]);

        return redirect()->route('admin.landingpage.menus.index')
                         ->with('success', 'Menu created successfully.');
    }

    /**
     * Display the specified resource.
     *
     * @param  \Modules\LandingPage\Entities\Menu  $menu
     * @return \Illuminate\Http\Response
     */
    public function show(Menu $menu)
    {
        // Typically, for menus, edit is more useful than show.
        // Redirect to edit or implement a detailed view if needed.
        return redirect()->route('admin.landingpage.menus.edit', $menu);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \Modules\LandingPage\Entities\Menu  $menu
     * @return \Illuminate\Contracts\View\View
     */
    public function edit(Menu $menu)
    {
        // Eager load items hierarchically, ensuring children are ordered correctly at all levels.
        $menu->load([
            'rootItems' => function ($query) {
                // Use 'sort_order' as per MenuItem model and migration
                $query->with('childrenRecursive')->orderBy('sort_order');
            },
            // Use 'sort_order' for items as well
            'items' => function ($query) { $query->orderBy('sort_order'); } 
        ]);
        $availablePages = LpPage::where('is_published', true)->orderBy('title')->pluck('title', 'id');
        $menuLocations = $this->getMenuLocations();

        // Define a list of available icons for the dropdown
        $availableIcons = [
            '' => '-- No Icon --',
            'fas fa-home' => 'Home (fas fa-home)',
            'fas fa-info-circle' => 'Info Circle (fas fa-info-circle)',
            'fas fa-envelope' => 'Envelope (fas fa-envelope)',
            'fas fa-user' => 'User (fas fa-user)',
            'fas fa-users' => 'Users (fas fa-users)',
            'fas fa-cog' => 'Settings/Cog (fas fa-cog)',
            'fas fa-briefcase' => 'Briefcase (fas fa-briefcase)',
            'fas fa-image' => 'Image (fas fa-image)',
            'fas fa-images' => 'Images (fas fa-images)',
            'fas fa-list' => 'List (fas fa-list)',
            'fas fa-newspaper' => 'Newspaper/Blog (fas fa-newspaper)',
            'fas fa-phone' => 'Phone (fas fa-phone)',
            'fas fa-link' => 'Link (fas fa-link)',
            'fas fa-external-link-alt' => 'External Link (fas fa-external-link-alt)',
            'fas fa-question-circle' => 'Question Circle (fas fa-question-circle)',
            'fas fa-cogs' => 'Cogs (fas fa-cogs)',
            'fas fa-tasks' => 'Tasks (fas fa-tasks)',
            'fas fa-folder-open' => 'Folder Open (fas fa-folder-open)',
            'fas fa-book' => 'Book (fas fa-book)',
            'fas fa-shopping-cart' => 'Shopping Cart (fas fa-shopping-cart)',
            'fas fa-tags' => 'Tags (fas fa-tags)',
            'fas fa-star' => 'Star (fas fa-star)',
            'fab fa-facebook-f' => 'Facebook (fab fa-facebook-f)',
            'fab fa-twitter' => 'Twitter (fab fa-twitter)',
            'fab fa-linkedin-in' => 'LinkedIn (fab fa-linkedin-in)',
            'fab fa-youtube' => 'YouTube (fab fa-youtube)',
            'fab fa-instagram' => 'Instagram (fab fa-instagram)',
            'fab fa-github' => 'GitHub (fab fa-github)',
            // Add more icons as needed
        ];

        return view('landingpage::admin.menus.edit', compact('menu', 'availablePages', 'menuLocations', 'availableIcons'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Modules\LandingPage\Entities\Menu  $menu
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Menu $menu)
    {
        $menuLocationsKeys = array_keys($this->getMenuLocations());
        $request->validate([
            'name' => 'required|string|max:255|unique:lp_menus,name,' . $menu->id,
            'location' => 'nullable|string|max:255|unique:lp_menus,location,' . $menu->id . '|in:' . implode(',', $menuLocationsKeys),
        ]);

        $menu->update([
            'name' => $request->input('name'),
            'location' => $request->input('location'),
        ]);

        return redirect()->route('admin.landingpage.menus.edit', $menu)
                         ->with('success', 'Menu updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \Modules\LandingPage\Entities\Menu  $menu
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Menu $menu)
    {
        // Consider what happens to menu items. The DB constraint will cascade delete them.
        // You might want to prevent deletion if it's a critical menu (e.g., 'main-navigation').
        // if ($menu->slug === 'main-navigation') {
        //     return redirect()->route('admin.landingpage.menus.index')
        //                      ->with('error', 'The main navigation menu cannot be deleted.');
        // }

        $menu->delete();

        return redirect()->route('admin.landingpage.menus.index')
                         ->with('success', 'Menu deleted successfully.');
    }
}