<?php

namespace Modules\DigitalVocano\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\DigitalVocano\Entities\Project;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class ProjectController extends Controller
{
    public function index()
    {
        $projects = Project::orderBy('sort_order')->paginate(15);
        return view('digitalvocano::admin.projects.index', compact('projects'));
    }

    public function create()
    {
        return view('digitalvocano::admin.projects.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:dv_projects,slug',
            'short_description' => 'nullable|string',
            'full_description' => 'nullable|string',
            'featured_image_file' => 'nullable|image|mimes:jpg,jpeg,png,gif,svg,webp|max:2048',
            'store_url' => 'nullable|url',
            'demo_url' => 'nullable|url',
            // 'documentation_url' => 'nullable|url', // Removed
            'internal_documentation_content' => 'nullable|string',
            'is_published' => 'boolean', // Handled by has()
            'sort_order' => 'nullable|integer',
        ]);

        $data = $request->except(['featured_image_file', '_token', '_method', 'is_published']);
        $data['is_published'] = $request->has('is_published');
        $data['slug'] = $request->input('slug') ?: Str::slug($request->input('name'));
        $data['sort_order'] = $request->input('sort_order', 0);


        if ($request->hasFile('featured_image_file')) {
            // Store in 'digitalvocano/project_images' inside the 'public' disk
            $path = $request->file('featured_image_file')->store('digitalvocano/project_images', 'public');
            $data['featured_image'] = $path; // Store the relative path
        }

        Project::create($data);

        return redirect()->route('admin.digitalvocano.projects.index')->with('success', 'Project created successfully.');
    }

    public function show(Project $project)
    {
        return view('digitalvocano::admin.projects.show', compact('project'));
    }

    public function edit(Project $project)
    {
        return view('digitalvocano::admin.projects.edit', compact('project'));
    }

    public function update(Request $request, Project $project)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:dv_projects,slug,' . $project->id,
            'short_description' => 'nullable|string',
            'full_description' => 'nullable|string',
            'featured_image_file' => 'nullable|image|mimes:jpg,jpeg,png,gif,svg,webp|max:2048',
            'store_url' => 'nullable|url',
            'demo_url' => 'nullable|url',
            // 'documentation_url' => 'nullable|url', // Removed
            'internal_documentation_content' => 'nullable|string',
            'is_published' => 'boolean', // Handled by has()
            'sort_order' => 'nullable|integer',
        ]);

        $data = $request->except(['featured_image_file', '_token', '_method', 'is_published']);
        $data['is_published'] = $request->has('is_published');
        $data['slug'] = $request->input('slug') ?: Str::slug($request->input('name'));
        $data['sort_order'] = $request->input('sort_order', $project->sort_order);


        if ($request->hasFile('featured_image_file')) {
            // Delete old image if it exists
            if ($project->featured_image) {
                // Assuming $project->featured_image stores the path relative to the 'public' disk's root
                if (Storage::disk('public')->exists($project->featured_image)) {
                    Storage::disk('public')->delete($project->featured_image);
                }
            }
            $path = $request->file('featured_image_file')->store('digitalvocano/project_images', 'public');
            $data['featured_image'] = $path; // Store the relative path
        }

        $project->update($data);

        return redirect()->route('admin.digitalvocano.projects.index')->with('success', 'Project updated successfully.');
    }

    public function destroy(Project $project)
    {
        if ($project->featured_image) {
            if (Storage::disk('public')->exists($project->featured_image)) {
                Storage::disk('public')->delete($project->featured_image);
            }
        }
        $project->delete();
        return redirect()->route('admin.digitalvocano.projects.index')->with('success', 'Project deleted successfully.');
    }
}
