<?php

namespace Modules\Blog\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Modules\Blog\Entities\Post;
use Modules\Blog\Entities\BlogCategory;
use Modules\Blog\Entities\BlogTag;
use Illuminate\Support\Str;

class PostController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:manage blog');
    }

    public function index(Request $request)
    {
        $query = Post::with(['user', 'categories'])->latest();
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }
        $posts = $query->paginate(15);
        return view('blog::admin.posts.index', compact('posts'));
    }

    public function create()
    {
        $categories = BlogCategory::orderBy('name')->get();
        $tags = BlogTag::orderBy('name')->get();
        return view('blog::admin.posts.create', compact('categories', 'tags'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'slug' => 'nullable|string|max:255|unique:blog_posts,slug',
            'featured_image_file' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'is_published' => 'boolean',
            'published_at_date' => 'nullable|date_format:Y-m-d',
            'published_at_time' => 'nullable|date_format:H:i',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:255',
            'categories' => 'nullable|array',
            'categories.*' => 'exists:blog_categories,id',
            'tags' => 'nullable|string', // Comma-separated tags
        ]);

        $data = $request->only(['title', 'content', 'meta_title', 'meta_description', 'meta_keywords']);
        $data['user_id'] = Auth::id();
        $data['is_published'] = $request->has('is_published');

        if ($request->filled('published_at_date')) {
            $data['published_at'] = $request->published_at_date . ' ' . ($request->published_at_time ?: '00:00:00');
        } elseif ($data['is_published'] && empty($request->published_at_date)) {
            $data['published_at'] = now();
        }

        if ($request->hasFile('featured_image_file')) {
            $data['featured_image'] = $request->file('featured_image_file')->store('blog/featured_images', 'public');
        }

        $post = Post::create($data); // Slug will be auto-generated

        if ($request->has('categories')) {
            $post->categories()->sync($request->categories);
        }

        if ($request->filled('tags')) {
            $tagNames = array_map('trim', explode(',', $request->tags));
            $tagIds = [];
            foreach ($tagNames as $tagName) {
                if (!empty($tagName)) {
                    $tag = BlogTag::firstOrCreate(['name' => $tagName]);
                    $tagIds[] = $tag->id;
                }
            }
            $post->tags()->sync($tagIds);
        }

        return redirect()->route('admin.blog.posts.index')->with('success', 'Blog post created successfully.');
    }

    public function edit(Post $post)
    {
        $categories = BlogCategory::orderBy('name')->get();
        $tags = BlogTag::orderBy('name')->get(); // For suggestion, actual tags are on post
        $post->load('categories', 'tags');
        return view('blog::admin.posts.edit', compact('post', 'categories', 'tags'));
    }

    public function update(Request $request, Post $post)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'slug' => 'nullable|string|max:255|unique:blog_posts,slug,' . $post->id,
            'featured_image_file' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'is_published' => 'boolean',
            'published_at_date' => 'nullable|date_format:Y-m-d',
            'published_at_time' => 'nullable|date_format:H:i',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:255',
            'categories' => 'nullable|array',
            'categories.*' => 'exists:blog_categories,id',
            'tags' => 'nullable|string',
        ]);

        $data = $request->only(['title', 'content', 'meta_title', 'meta_description', 'meta_keywords']);
        $data['is_published'] = $request->has('is_published');

        if ($request->filled('published_at_date')) {
            $data['published_at'] = $request->published_at_date . ' ' . ($request->published_at_time ?: '00:00:00');
        } elseif ($data['is_published'] && !$post->published_at) {
            $data['published_at'] = now();
        } elseif (!$data['is_published']) {
            $data['published_at'] = null;
        }

        if ($request->hasFile('featured_image_file')) {
            if ($post->featured_image && Storage::disk('public')->exists($post->featured_image)) {
                Storage::disk('public')->delete($post->featured_image);
            }
            $data['featured_image'] = $request->file('featured_image_file')->store('blog/featured_images', 'public');
        }

        // Slug is handled by HasSlug trait on update if title changes and slug is not manually set
        if ($request->filled('slug') && $request->slug !== $post->slug) {
            $data['slug'] = $request->slug;
        }

        $post->update($data);

        $post->categories()->sync($request->input('categories', []));

        $tagIds = [];
        if ($request->filled('tags')) {
            $tagNames = array_map('trim', explode(',', $request->tags));
            foreach ($tagNames as $tagName) {
                if (!empty($tagName)) {
                    $tag = BlogTag::firstOrCreate(['name' => $tagName]);
                    $tagIds[] = $tag->id;
                }
            }
        }
        $post->tags()->sync($tagIds);

        return redirect()->route('admin.blog.posts.index')->with('success', 'Blog post updated successfully.');
    }

    public function destroy(Post $post)
    {
        if ($post->featured_image && Storage::disk('public')->exists($post->featured_image)) {
            Storage::disk('public')->delete($post->featured_image);
        }
        $post->delete();
        return redirect()->route('admin.blog.posts.index')->with('success', 'Blog post deleted successfully.');
    }
}