<?php

namespace Modules\AuthorizeNetGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Modules\AuthorizeNetGateway\Services\AuthorizeNetService;
use App\Services\CreditService;
use App\Services\WalletService;
use Spatie\Permission\Models\Role;

class AuthorizeNetSubscriptionController extends Controller
{
    protected AuthorizeNetService $authorizeNetService;
    protected CreditService $creditService;
    protected WalletService $walletService;

    public function __construct(
        AuthorizeNetService $authorizeNetService,
        CreditService $creditService,
        WalletService $walletService
    ) {
        $this->authorizeNetService = $authorizeNetService;
        $this->creditService = $creditService;
        $this->walletService = $walletService;
    }

    public function showCheckoutForm(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        if (setting('authorizenet_enabled', '0') != '1') {
            return redirect()->route('subscription.plans')->with('error', 'Authorize.Net payments are currently disabled.');
        }

        $apiLoginId = setting('authorizenet_login_id');
        $publicClientKey = setting('authorizenet_public_client_key');
        
        if (!$apiLoginId || !$publicClientKey) {
            Log::error("Authorize.Net checkout form: API Login ID or Public Client Key is not set.");
            return redirect()->route('subscription.plans')->with('error', 'Authorize.Net gateway is not configured correctly.');
        }

        return view('authorizenetgateway::subscriptions.checkout', ['plan' => $subscriptionPlan, 'apiLoginId' => $apiLoginId, 'publicClientKey' => $publicClientKey]);
    }

    public function processSubscriptionWithAcceptJs(Request $request, SubscriptionPlan $plan)
    {
        if (setting('authorizenet_enabled', '0') != '1') {
            return redirect()->route('subscription.plans')->with('error', 'Authorize.Net payments are currently disabled.');
        }

        $request->validate([
            'dataDescriptor' => 'required|string', // Matches form field name from checkout.blade.php
            'dataValue' => 'required|string',      // Matches form field name from checkout.blade.php
            // Add validation for billing details if you plan to use them directly from the request
            // For Accept.js, these are primarily for tokenization and AVS,
            // but it's good to have them if you also store/process them.
            'billToFirstName' => 'sometimes|required|string|max:50',
            'billToLastName' => 'sometimes|required|string|max:50',
            'billToAddress' => 'sometimes|required|string|max:60',
            'billToCity' => 'sometimes|required|string|max:40',
            'billToState' => 'sometimes|required|string|max:40', // Or use a validation rule for valid states/provinces
            'billToZip' => 'sometimes|required|string|max:20',
            'billToCountry' => 'sometimes|required|string|max:60', // Or use a validation rule for valid countries
        ]);

        /** @var User $user */
        $user = Auth::user();

        try {
            $opaqueData = [
                'dataDescriptor' => $request->input('dataDescriptor'),
                'dataValue' => $request->input('dataValue'),
            ];

            // Use billing details from the request if provided, otherwise fallback to user's profile
            // This assumes your checkout.blade.php submits these fields.
            // If Accept.js handles billing details entirely for tokenization,
            // you might only need basic customer info here.
            // However, sending full billing details to Authorize.Net API is best for AVS.

            $customerDetails = [
                'email' => $user->email,
                'firstName' => $request->input('billToFirstName', $user->first_name ?? (explode(' ', $user->name)[0] ?? '')),
                'lastName' => $request->input('billToLastName', $user->last_name ?? (explode(' ', $user->name, 2)[1] ?? '')),
                'address' => $request->input('billToAddress', $user->billing_address ?? null),
                'city' => $request->input('billToCity', $user->billing_city ?? null),
                'state' => $request->input('billToState', $user->billing_state ?? null),
                'zip' => $request->input('billToZip', $user->billing_zip ?? null),
                'country' => $request->input('billToCountry', $user->billing_country ?? 'USA'),
            ];

            // Ensure required fields for Authorize.Net are not null, even if empty strings
            // Authorize.Net schema might be strict about presence vs. null.
            foreach (['firstName', 'lastName', 'address', 'city', 'state', 'zip', 'country'] as $key) {
                if (is_null($customerDetails[$key])) {
                    $customerDetails[$key] = ''; // Send empty string instead of null if not provided
                }
            }

            $invoiceNumber = 'SUB-' . Str::random(6) . '-' . $user->id;
            $description = "Subscription to {$plan->name}";

            $response = $this->authorizeNetService->chargeCreditCard($plan->price, $opaqueData, $customerDetails, $invoiceNumber, $description);

            if ($response != null && $response->getMessages()->getResultCode() == "Ok" && $response->getTransactionResponse() != null && $response->getTransactionResponse()->getMessages() != null) {
                $transactionId = $response->getTransactionResponse()->getTransId();

                // Deactivate other existing subscriptions for this user
                $user->subscriptions()
                    ->whereIn('status', ['active', 'trialing'])
                    ->update(['status' => 'cancelled', 'ends_at' => now(), 'cancelled_at' => now()]);

                // Create new local subscription record
                $subscription = Subscription::create([
                    'user_id' => $user->id,
                    'subscription_plan_id' => $plan->id,
                    'payment_gateway' => 'authorizenetgateway',
                    'gateway_transaction_id' => $transactionId,
                    'status' => $plan->trial_period_days > 0 ? 'trialing' : 'active',
                    'price_at_purchase' => $plan->price,
                    'currency_at_purchase' => $plan->currency,
                    'trial_ends_at' => $plan->trial_period_days > 0 ? now()->addDays($plan->trial_period_days) : null,
                    'starts_at' => now(),
                    'ends_at' => now()->add($plan->interval, $plan->interval_count),
                ]);

                // Award credits
                if (function_exists('setting') && setting('credits_system_enabled', '0') == '1' && $plan->credits_awarded_on_purchase > 0) {
                    $this->creditService->awardCredits($user, $plan->credits_awarded_on_purchase, 'award_subscription_purchase', "Credits for {$plan->name} subscription", $subscription);
                }

                // Assign target role
                if (!empty($plan->target_role) && class_exists(Role::class) && Role::where('name', $plan->target_role)->where('guard_name', 'web')->exists()) {
                    $user->syncRoles([$plan->target_role]);
                }

                return redirect()->route('dashboard')->with('success', "Successfully subscribed to {$plan->name} via Authorize.Net!");
            }
            // Use the new getErrorMessage method from AuthorizeNetService
            $errorMessage = 'Payment failed.';
            if (method_exists($this->authorizeNetService, 'getErrorMessage')) {
                $errorMessage = $this->authorizeNetService->getErrorMessage($response);
            }
            Log::error("Authorize.Net Subscription Error for user {$user->id}, plan {$plan->id}: " . $errorMessage, [
                'response_raw' => $response ? $response->jsonSerialize() : null, // Log the full response object if available
                'customer_details_sent' => $customerDetails,
                'opaque_data_descriptor' => $opaqueData['dataDescriptor']
            ]);
            return redirect()->route('subscription.authorizenet.checkout', ['subscriptionPlan' => $plan->slug])->with('error', 'Authorize.Net payment failed: ' . $errorMessage)->withInput();
        } catch (\Exception $e) {
            Log::error("Authorize.Net Subscription Exception for user {$user->id}, plan {$plan->id}: " . $e->getMessage());
            return redirect()->route('subscription.authorizenet.checkout', ['subscriptionPlan' => $plan->slug])->with('error', 'An error occurred while processing your payment with Authorize.Net.')->withInput();
        }
    }

    // --- Wallet Deposit Methods ---

    public function showWalletDepositForm(Request $request)
    {
        if (setting('authorizenet_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Authorize.Net deposits are currently disabled.');
        }

        $request->validate(['amount' => 'required|numeric|min:1']);
        $amount = (float) $request->input('amount');

        $apiLoginId = setting('authorizenet_login_id');
        $publicClientKey = setting('authorizenet_public_client_key');

        if (!$apiLoginId || !$publicClientKey) {
            Log::error("Authorize.Net wallet deposit form: API Login ID or Public Client Key is not set.");
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Authorize.Net gateway is not configured correctly.');
        }

        return view('authorizenetgateway::wallet.deposit_form', compact('apiLoginId', 'publicClientKey', 'amount'));
    }

    public function processWalletDepositWithAcceptJs(Request $request)
    {
        if (setting('authorizenet_enabled', '0') != '1' || setting('allow_wallet_deposits', '0') != '1') {
            return redirect()->route('user.wallet.deposit.form')->with('error', 'Authorize.Net deposits are currently disabled.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:1',
            'authorizenet_data_descriptor' => 'required|string',
            'authorizenet_data_value' => 'required|string', // Payment nonce
            'billToFirstName' => 'sometimes|required|string|max:50',
            'billToLastName' => 'sometimes|required|string|max:50',
            'billToAddress' => 'sometimes|required|string|max:60',
            'billToCity' => 'sometimes|required|string|max:40',
            'billToState' => 'sometimes|required|string|max:40',
            'billToZip' => 'sometimes|required|string|max:20',
            'billToCountry' => 'sometimes|required|string|max:60',
        ]);

        /** @var User $user */
        $user = Auth::user();
        $amount = (float) $request->input('amount');
        $currency = strtoupper(setting('wallet_default_currency', 'USD'));

        try {
            $opaqueData = [
                'dataDescriptor' => $request->input('authorizenet_data_descriptor'),
                'dataValue' => $request->input('authorizenet_data_value'),
            ];

            $customerDetails = [
                'email' => $user->email,
                'firstName' => $request->input('billToFirstName', $user->first_name ?? (explode(' ', $user->name)[0] ?? '')),
                'lastName' => $request->input('billToLastName', $user->last_name ?? (explode(' ', $user->name, 2)[1] ?? '')),
                'address' => $request->input('billToAddress', $user->billing_address ?? null),
                'city' => $request->input('billToCity', $user->billing_city ?? null),
                'state' => $request->input('billToState', $user->billing_state ?? null),
                'zip' => $request->input('billToZip', $user->billing_zip ?? null),
                'country' => $request->input('billToCountry', $user->billing_country ?? 'USA'),
            ];
            foreach (['firstName', 'lastName', 'address', 'city', 'state', 'zip', 'country'] as $key) {
                if (is_null($customerDetails[$key])) {
                    $customerDetails[$key] = '';
                }
            }

            $invoiceNumber = 'DEPOSIT-' . Str::random(6) . '-' . $user->id;
            $description = "Wallet Deposit for user {$user->id}";

            $response = $this->authorizeNetService->chargeCreditCard($amount, $opaqueData, $customerDetails, $invoiceNumber, $description);

            if ($response != null && $response->getMessages()->getResultCode() == "Ok" && $response->getTransactionResponse() != null && $response->getTransactionResponse()->getMessages() != null) {
                $transactionId = $response->getTransactionResponse()->getTransId();
                $this->walletService->deposit($user, $amount, $currency, 'authorizenetgateway', $transactionId, "Wallet deposit via Authorize.Net");
                return redirect()->route('user.wallet.history')->with('success', 'Successfully deposited ' . $currency . ' ' . number_format($amount, 2) . ' to your wallet.');
            }
            // Use the new getErrorMessage method from AuthorizeNetService
            $errorMessage = 'Payment failed.';
            if (method_exists($this->authorizeNetService, 'getErrorMessage')) {
                $errorMessage = $this->authorizeNetService->getErrorMessage($response);
            }
            Log::error("Authorize.Net Wallet Deposit Error for user {$user->id}: " . $errorMessage, [
                'response_raw' => $response ? $response->jsonSerialize() : null,
                'customer_details_sent' => $customerDetails,
                'opaque_data_descriptor' => $opaqueData['dataDescriptor']
            ]);
            return redirect()->route('wallet.authorizenet.depositForm', ['amount' => $amount])->with('error', 'Authorize.Net payment failed: ' . $errorMessage)->withInput();
        } catch (\Exception $e) {
            Log::error("Authorize.Net Wallet Deposit Exception for user {$user->id}: " . $e->getMessage());
            return redirect()->route('wallet.authorizenet.depositForm', ['amount' => $amount])->with('error', 'An error occurred while processing your deposit with Authorize.Net.')->withInput();
        }
    }
}