<?php

namespace Modules\AppManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Modules\AppManager\Entities\ManagedScript;
use Modules\AppManager\Entities\License;
use Modules\AppManager\Entities\DownloadToken;
use Modules\AppManager\Entities\DownloadableFile;
    use Modules\AppManager\Entities\ActivationLog; // Import ActivationLog
use Tests\TestCase; // Your base TestCase from the main application

class AppManagerApiTest extends TestCase
{
    use RefreshDatabase;

    protected ManagedScript $script;
    protected License $license;

    protected function setUp(): void
    {
        parent::setUp();

        // Common setup for most tests
        $this->script = ManagedScript::factory()->create([
            'slug' => 'test-script',
            'current_version' => '1.0.0',
            'envato_item_id' => '12345678' // Example Envato ID
        ]);

        $this->license = License::factory()->create([
            'managed_script_id' => $this->script->id,
            'license_key' => 'VALID-LICENSE-KEY',
            'status' => 'active',
            'activation_limit' => 1,
            'current_activations' => 0,
        ]);
    }

    /**
     * @test
     */
    public function can_activate_with_valid_license_key(): void
    {
        $payload = [
            'product_slug' => $this->script->slug,
            'domain' => 'https://client.example.com',
            'license_key' => $this->license->license_key,
        ];

        $response = $this->postJson(route('api.appmanager.activate'), $payload);

        $response->assertStatus(200)
            ->assertJson([
                'status' => 'success',
                'message' => 'Application activated successfully.',
            ])
            ->assertJsonStructure([
                'status', 'message', 'activation_token',
                'product_info' => ['name', 'version'],
                'license_info' => ['status', 'expires_at', 'supported_until', 'type'],
            ]);

        $this->assertDatabaseHas('am_licenses', [
            'id' => $this->license->id,
            'current_activations' => 1,
        ]);
        $this->assertDatabaseHas('am_activation_logs', [
            'license_id' => $this->license->id,
            'activated_domain' => 'https://client.example.com',
            'status' => 'success',
        ]);
        $this->assertDatabaseCount('am_download_tokens', 1);
    }

    /**
     * @test
     */
    public function can_activate_with_valid_purchase_code_and_mocked_envato_success(): void
    {
        Http::fake([
            'api.envato.com/*' => Http::sequence()
                ->push([
                    'item' => ['id' => $this->script->envato_item_id],
                    'buyer' => 'testbuyer@example.com',
                    'supported_until' => now()->addYear()->toIso8601String(),
                    'license' => 'Regular License'
                ], 200)
        ]);

        $payload = [
            'product_slug' => $this->script->slug,
            'domain' => 'https://client-envato.example.com',
            'purchase_code' => 'VALID-PURCHASE-CODE-FORMAT-12345', // Format doesn't matter for mock
        ];

        $response = $this->postJson(route('api.appmanager.activate'), $payload);

        $response->assertStatus(200)
            ->assertJsonPath('status', 'success');

        $this->assertDatabaseHas('am_licenses', [
            'managed_script_id' => $this->script->id,
            'purchase_code' => 'VALID-PURCHASE-CODE-FORMAT-12345',
            'type' => 'codecanyon',
            'current_activations' => 1,
        ]);
        $this->assertDatabaseHas('am_activation_logs', [
            'purchase_code_attempt' => 'VALID-PURCHASE-CODE-FORMAT-12345',
            'activated_domain' => 'https://client-envato.example.com',
            'status' => 'success',
        ]);
    }


    /**
     * @test
     */
    public function activate_fails_if_license_not_found(): void
    {
        $payload = [
            'product_slug' => $this->script->slug,
            'domain' => 'https://client.example.com',
            'license_key' => 'NON-EXISTENT-KEY',
        ];

        $response = $this->postJson(route('api.appmanager.activate'), $payload);

        $response->assertStatus(403)
            ->assertJson(['status' => 'error', 'message' => 'License key or purchase code not found or invalid for this product.']);
    }

    /**
     * @test
     */
    public function activate_fails_if_activation_limit_reached(): void
    {
        $this->license->update(['current_activations' => 1, 'activation_limit' => 1]);

        $payload = [
            'product_slug' => $this->script->slug,
            'domain' => 'https://another-client.example.com',
            'license_key' => $this->license->license_key,
        ];

        $response = $this->postJson(route('api.appmanager.activate'), $payload);

        $response->assertStatus(403)
            ->assertJson(['status' => 'error', 'message' => 'Activation limit reached for this license.']);
    }

    /**
     * @test
     */
    public function activate_fails_with_invalid_payload(): void
    {
        $response = $this->postJson(route('api.appmanager.activate'), []);
        $response->assertStatus(400)
                 ->assertJsonValidationErrors(['product_slug', 'domain']);
    }

    /**
     * @test
     */
    public function can_validate_license_successfully(): void
    {
        // Simulate a successful activation first
        ActivationLog::factory()->create([
            'license_id' => $this->license->id,
            'managed_script_id' => $this->script->id,
            'activated_domain' => 'https://client.example.com',
            'status' => 'success',
        ]);

        $payload = [
            'license_key' => $this->license->license_key,
            'product_slug' => $this->script->slug,
            'domain' => 'https://client.example.com',
        ];

        $response = $this->postJson(route('api.appmanager.validate'), $payload);

        $response->assertStatus(200)
            ->assertJson(['status' => 'valid', 'message' => 'License is active.']);
    }

    /**
     * @test
     */
    public function validate_license_fails_if_not_active_for_domain(): void
    {
        $payload = [
            'license_key' => $this->license->license_key,
            'product_slug' => $this->script->slug,
            'domain' => 'https://unknown-domain.example.com', // Not activated for this domain
        ];

        $response = $this->postJson(route('api.appmanager.validate'), $payload);

        $response->assertStatus(403)
            ->assertJson(['status' => 'invalid', 'message' => 'License not activated for this domain.']);
    }

    /**
     * @test
     */
    public function can_download_file_with_valid_token(): void
    {
        Storage::fake(config('appmanager.storage_disk')); // Use fake storage

        $downloadableFile = DownloadableFile::factory()->create([
            'managed_script_id' => $this->script->id,
            'file_name' => 'test_file.zip',
            'file_path' => 'app_manager_files/' . $this->script->slug . '/1.0.0/test_file.zip',
            'version' => '1.0.0',
            'target_path' => 'app/test_file.zip',
        ]);
        // Create a dummy file in fake storage
        Storage::disk(config('appmanager.storage_disk'))->put($downloadableFile->file_path, 'dummy content');

        $downloadToken = DownloadToken::factory()->create([
            'license_id' => $this->license->id,
            'expires_at' => now()->addHour(),
            'max_uses' => 1,
        ]);

        $payload = [
            'activation_token' => $downloadToken->token,
            'product_slug' => $this->script->slug,
            'file_identifier' => $downloadableFile->target_path, // Assuming identifier is target_path
            'version' => $downloadableFile->version,
        ];

        $response = $this->postJson(route('api.appmanager.download.file'), $payload);

        $response->assertStatus(200)
                 ->assertHeader('Content-Disposition', 'attachment; filename="' . $downloadableFile->file_name . '"');
        $this->assertEquals('dummy content', $response->streamedContent());

        $this->assertDatabaseHas('am_download_tokens', [
            'id' => $downloadToken->id,
            'uses' => 1,
        ]);
    }

    /**
     * @test
     */
    public function download_file_fails_with_invalid_token(): void
    {
        $payload = [
            'activation_token' => 'INVALID-TOKEN',
            'product_slug' => $this->script->slug,
            'file_identifier' => 'app/some_file.zip',
            'version' => '1.0.0',
        ];

        $response = $this->postJson(route('api.appmanager.download.file'), $payload);
        $response->assertStatus(403); // Or 404 depending on controller logic
    }

    /**
     * @test
     */
    public function can_validate_activation_token_successfully(): void
    {
        $downloadToken = DownloadToken::factory()->create([
            'license_id' => $this->license->id,
            'expires_at' => now()->addHour(),
        ]);
        // Simulate an activation for the domain associated with this license
        ActivationLog::factory()->create([
            'license_id' => $this->license->id,
            'managed_script_id' => $this->script->id,
            'activated_domain' => 'https://client.example.com',
            'status' => 'success',
        ]);


        $payload = [
            'activation_token' => $downloadToken->token,
            'domain' => 'https://client.example.com',
            'product_slug' => $this->script->slug,
        ];

        $response = $this->postJson(route('api.appmanager.validate.activation'), $payload);

        $response->assertStatus(200)
            ->assertJson(['status' => 'success', 'is_valid' => true]);
    }

    /**
     * @test
     */
    public function validate_activation_token_fails_if_domain_mismatch(): void
    {
        $downloadToken = DownloadToken::factory()->create([
            'license_id' => $this->license->id,
            'expires_at' => now()->addHour(),
        ]);
        // Simulate an activation for a DIFFERENT domain
        ActivationLog::factory()->create([
            'license_id' => $this->license->id,
            'managed_script_id' => $this->script->id,
            'activated_domain' => 'https://original-activation.example.com',
            'status' => 'success',
        ]);

        $payload = [
            'activation_token' => $downloadToken->token,
            'domain' => 'https://different-client.example.com', // Mismatched domain
            'product_slug' => $this->script->slug,
        ];

        $response = $this->postJson(route('api.appmanager.validate.activation'), $payload);

        $response->assertStatus(403) // Or appropriate error code
            ->assertJson(['status' => 'error', 'is_valid' => false]);
    }
}
