<?php

namespace Modules\WebPilotAI\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\User; // Assuming User model is in App\Models
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Modules\WebPilotAI\Models\AITemplate; // Import AITemplate model

class Website extends Model
{
    use HasFactory;

    // Status Constants
    public const STATUS_PENDING_GENERATION = 'pending_generation';
    public const STATUS_GENERATING = 'generating';
    public const STATUS_COMPLETED = 'completed';
    public const STATUS_FAILED = 'failed';
    public const STATUS_PENDING_REGENERATION = 'pending_regeneration';
    public const STATUS_DEPLOYING = 'deploying';
    public const STATUS_DEPLOYED = 'deployed';
    public const STATUS_DEPLOYMENT_FAILED = 'deployment_failed';


    protected $table = 'websites';

    protected $fillable = [
        'user_id',
        'name',
        'description_prompt',
        'ai_model_id',
        'ai_template_id',
        'pages_structure',
        'status',
        'generated_content_path',
        'last_generated_at',
        'generation_error',
        'deployment_details',
    ];

    protected $casts = [
        'last_generated_at' => 'datetime',
        'deployment_details' => 'array',
        'pages_structure' => 'array', // Automatically cast to/from JSON
    ];

    /**
     * Get the user that owns the website.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the AI model used for generating this website.
     */
    public function aiModel(): BelongsTo
    {
        return $this->belongsTo(AIModel::class, 'ai_model_id');
    }

    /**
     * Get the AI template associated with the website.
     */
    public function aiTemplate(): BelongsTo
    {
        return $this->belongsTo(AITemplate::class, 'ai_template_id');
    }

    /**
     * The AI style presets that belong to the website.
     */
    public function aiStylePresets(): BelongsToMany
    {
        return $this->belongsToMany(AIStylePreset::class, 'ai_style_preset_website', 'website_id', 'ai_style_preset_id');
    }

    /**
     * Get the assets associated with the website.
     */
    public function assets(): HasMany
    {
        return $this->hasMany(WebsiteAsset::class);
    }

    // TODO: Add factory if needed: protected static function newFactory() { return \Modules\WebPilotAI\Database\factories\WebsiteFactory::new(); }
}