<?php

namespace Modules\WebPilotAI\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\WebPilotAI\Models\AIModel; // Import the AIModel
use Illuminate\Http\RedirectResponse; // Added for type hinting
use App\Models\Setting; // Import the Setting model
use Illuminate\Support\Facades\Validator; // For custom validation logic if needed
use Illuminate\Validation\Rule; // For Rule::in validation
use Illuminate\Support\Facades\Log; // Import Log facade
use Nwidart\Modules\Facades\Module; // To access module.json settings

class AIModelController extends Controller
{
    // Helper method to get provider data.
    // In a real application, this might come from a config file or a dedicated service.
    private function getProvidersData(): array
    {
        // Example: Fetching from a hypothetical config file: config('webpilotai.providers')
        // For now, keeping it similar to your existing structure for demonstration
        return [
            'OpenAI' => ['identifiers' => ['gpt-4-turbo', 'gpt-4o', 'gpt-3.5-turbo']],
            'Anthropic' => ['identifiers' => ['claude-3-opus-20240229', 'claude-3-sonnet-20240229', 'claude-3-haiku-20240307']],
            'GoogleAI' => ['identifiers' => ['gemini-pro', 'gemini-1.5-pro-latest', 'gemini-1.5-flash-latest']],
            // Add other providers here
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Fetch models from the database with pagination, ordered by name
        $aiModels = AIModel::orderBy('name')->paginate(15); // Use paginate for pagination

        return view('webpilotai::admin.aimodels.index', compact('aiModels')); // Pass $aiModels
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $providersData = $this->getProvidersData();

        // Fetch API key settings defined in WebPilotAI's module.json
        $apiKeySettings = [];
        $webPilotAIModule = Module::find('WebPilotAI');
        if ($webPilotAIModule) {
            $moduleDefinedSettings = $webPilotAIModule->get('settings');
            if (is_array($moduleDefinedSettings)) {
                foreach ($moduleDefinedSettings as $setting) {
                    // Assuming API key settings are grouped or have a specific characteristic
                    // For example, if they are in 'AI Model Providers' group and are passwords
                    if (isset($setting['group']) && $setting['group'] === 'AI Model Providers' && isset($setting['type']) && $setting['type'] === 'password') {
                        // The key in module.json is like 'openai_api_key'.
                        // The full setting key in the database is 'webpilotai.openai_api_key'.
                        $fullSettingKey = 'webpilotai.' . $setting['key'];
                        $apiKeySettings[$fullSettingKey] = $setting['label'] . " (Key: " . $setting['key'] . ")";
                    }
                }
            }
        }
        ksort($apiKeySettings);

        // Ensure $apiKeySettings is an associative array for the view
        return view('webpilotai::admin.aimodels.create', compact('providersData', 'apiKeySettings'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
         $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'identifier' => 'required|string|max:255|unique:ai_models,identifier', // Ensures identifier is unique
            'provider' => ['required', 'string', 'max:100', Rule::in(array_keys($this->getProvidersData()))],
            'api_key_setting_name' => 'required|string|max:255', // Should be required if a provider needs an API key
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_premium' => 'nullable|boolean',
        ]);

        // Adjust 'is_active' and 'is_premium' as checkboxes send '1' or nothing
        $validatedData['is_active'] = $request->has('is_active');
        $validatedData['is_premium'] = $request->has('is_premium');

        AIModel::create($validatedData);

        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model added successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \Modules\WebPilotAI\Models\AIModel  $model
     * @return \Illuminate\View\View
     */
    public function edit(AIModel $model) // Using Route Model Binding
    {
        $providersData = $this->getProvidersData();

        // Fetch API key settings defined in WebPilotAI's module.json (same as create)
        $apiKeySettings = [];
        $webPilotAIModule = Module::find('WebPilotAI');
        if ($webPilotAIModule) {
            $moduleDefinedSettings = $webPilotAIModule->get('settings');
            if (is_array($moduleDefinedSettings)) {
                foreach ($moduleDefinedSettings as $setting) {
                    if (isset($setting['group']) && $setting['group'] === 'AI Model Providers' && isset($setting['type']) && $setting['type'] === 'password') {
                        $fullSettingKey = 'webpilotai.' . $setting['key'];
                        $apiKeySettings[$fullSettingKey] = $setting['label'] . " (Key: " . $setting['key'] . ")";
                    }
                }
            }
        }
        ksort($apiKeySettings);

        // If the currently saved api_key_setting_name for the model is not in the list
        // (e.g., if a setting was removed from module.json), add it to the list so it can still be selected.
        // This ensures the previously selected value is available in the dropdown.
        if ($model->api_key_setting_name && !isset($apiKeySettings[$model->api_key_setting_name])) {
            // Attempt to find the label from the settings table if it exists there
            $existingSetting = Setting::where('key', $model->api_key_setting_name)->first();
            $label = $existingSetting ? $existingSetting->name : $model->api_key_setting_name;
            $apiKeySettings[$model->api_key_setting_name] = $label . " (Currently Selected - Potentially Obsolete)";
            // Re-sort if you want the obsolete key to appear in alphabetical order, or leave it to appear at the end.
            // ksort($apiKeySettings);
        }

        // Ensure $apiKeySettings is an associative array for the view
        return view('webpilotai::admin.aimodels.edit', compact('model', 'providersData', 'apiKeySettings'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Illuminate\Http\Request  $request
     * @param  \Modules\WebPilotAI\Models\AIModel  $model
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, AIModel $model): RedirectResponse // Using Route Model Binding
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255', // Keep name as text input
            'identifier' => ['required', 'string', 'max:255', Rule::unique('ai_models', 'identifier')->ignore($model->id)],
            'provider' => ['required', 'string', 'max:100', Rule::in(array_keys($this->getProvidersData()))],
            'api_key_setting_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_premium' => 'nullable|boolean',
        ]);

        $validatedData['is_active'] = $request->has('is_active');
        $validatedData['is_premium'] = $request->has('is_premium');

        $model->update($validatedData);

        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \Modules\WebPilotAI\Models\AIModel  $model
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(AIModel $model): RedirectResponse // Using Route Model Binding
    {
        $model->delete();
        return redirect()->route('admin.webpilotai.models.index')->with('success', 'AI Model deleted successfully.');
    }
}
