<?php

namespace Modules\StripeGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\StripeGateway\Services\StripeService;
use Stripe\Event as StripeEvent;
use Stripe\Subscription as StripeSubscriptionObject; // Alias to avoid conflict

class StripeWebhookController extends Controller
{
    protected StripeService $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    public function handleWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $event = null;

        try {
            $event = $this->stripeService->verifyWebhookSignature($payload, $sigHeader);
        } catch (\UnexpectedValueException $e) {
            // Invalid payload
            Log::error('Stripe Webhook Error: Invalid payload.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            Log::error('Stripe Webhook Error: Invalid signature.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Invalid signature'], 400);
        } catch (\Exception $e) {
            Log::error('Stripe Webhook Error: Could not verify signature.', ['exception' => $e->getMessage()]);
            return response()->json(['error' => 'Signature verification error'], 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'checkout.session.completed':
                $session = $event->data->object; // contains a \Stripe\Checkout\Session
                // This is often handled by the success redirect, but good to have a fallback
                // or if you need to do something specific when checkout is fully completed.
                // Ensure you get the subscription ID from the session: $session->subscription
                Log::info('Stripe Webhook: checkout.session.completed', ['session_id' => $session->id, 'subscription_id' => $session->subscription]);
                if ($session->subscription && $session->payment_status == 'paid') {
                    $this->handleSubscriptionUpdate($session->subscription, $session->customer);
                }
                break;

            case 'customer.subscription.created':
            case 'customer.subscription.updated':
            case 'customer.subscription.resumed': // Stripe specific for resumed subscriptions
                $stripeSubscription = $event->data->object; // contains a \Stripe\Subscription
                Log::info('Stripe Webhook: ' . $event->type, ['subscription_id' => $stripeSubscription->id]);
                $this->handleSubscriptionUpdate($stripeSubscription->id, $stripeSubscription->customer);
                break;

            case 'customer.subscription.trial_will_end':
                $stripeSubscription = $event->data->object;
                Log::info('Stripe Webhook: customer.subscription.trial_will_end', ['subscription_id' => $stripeSubscription->id]);
                // Send a notification to the user
                break;

            case 'customer.subscription.deleted': // Occurs when a subscription is canceled immediately or at period end.
                $stripeSubscription = $event->data->object; // contains a \Stripe\Subscription
                Log::info('Stripe Webhook: customer.subscription.deleted', ['subscription_id' => $stripeSubscription->id]);
                $this->handleSubscriptionCancellation($stripeSubscription->id);
                break;

            case 'invoice.payment_succeeded':
                $invoice = $event->data->object; // contains an \Stripe\Invoice
                Log::info('Stripe Webhook: invoice.payment_succeeded', ['invoice_id' => $invoice->id, 'subscription_id' => $invoice->subscription]);
                if ($invoice->subscription) {
                    // This event confirms a recurring payment was successful.
                    // The customer.subscription.updated event usually handles the date changes.
                    $this->handleSubscriptionUpdate($invoice->subscription, $invoice->customer);
                }
                break;

            case 'invoice.payment_failed':
                $invoice = $event->data->object; // contains an \Stripe\Invoice
                Log::error('Stripe Webhook: invoice.payment_failed', ['invoice_id' => $invoice->id, 'subscription_id' => $invoice->subscription]);
                if ($invoice->subscription) {
                    $this->handleSubscriptionPaymentFailure($invoice->subscription);
                }
                break;
            
            // ... handle other event types
            default:
                Log::info('Stripe Webhook: Received unhandled event type ' . $event->type);
        }

        return response()->json(['status' => 'success']);
    }

    protected function handleSubscriptionUpdate($stripeSubscriptionId, $stripeCustomerId)
    {
        try {
            if (!setting('stripe_secret_key')) {
                 throw new \Exception('Stripe secret key is not configured for webhook processing.');
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            $stripeSubscription = StripeSubscriptionObject::retrieve($stripeSubscriptionId); // Use aliased StripeSubscriptionObject

            $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscription->id)->first();

            $user = User::where('stripe_customer_id', $stripeCustomerId)->first();
            if (!$user && $localSubscription) { // Fallback if customer ID wasn't set on user yet
                $user = $localSubscription->user;
            }
            
            if (!$user) {
                Log::error("Stripe Webhook: User not found for Stripe Customer ID: {$stripeCustomerId}");
                return;
            }
            
            // Ensure user has stripe_customer_id if it's missing
            if ($user && $stripeCustomerId && !$user->stripe_customer_id) {
                $user->stripe_customer_id = $stripeCustomerId;
                $user->save();
            }

            $plan = null;
            if (isset($stripeSubscription->items->data[0]->price->product)) {
                // This assumes you store Stripe Product ID or a mapping if your plan slugs don't match Stripe Product IDs
                // For simplicity, let's assume plan_id was stored in metadata or you can find it
                // A more robust way is to store Stripe Price ID on your SubscriptionPlan model.
                // For now, we'll rely on existing local subscription or metadata if available.
            }
            if (!$plan && $localSubscription) {
                $plan = $localSubscription->plan;
            }
            // If creating a new subscription from webhook (e.g. checkout.session.completed without success redirect hitting first)
            // You'd need to get plan_id from $stripeSubscription->metadata or items.
            // For now, this primarily updates existing subscriptions.

            if ($localSubscription) {
                $localSubscription->status = $stripeSubscription->status;
                $localSubscription->starts_at = \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_start);
                $localSubscription->ends_at = \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end);
                $localSubscription->trial_ends_at = $stripeSubscription->trial_end ? \Carbon\Carbon::createFromTimestamp($stripeSubscription->trial_end) : null;
                $localSubscription->cancelled_at = $stripeSubscription->cancel_at_period_end ? ($stripeSubscription->canceled_at ? \Carbon\Carbon::createFromTimestamp($stripeSubscription->canceled_at) : now()) : null;
                $localSubscription->save();
                Log::info("Stripe Webhook: Updated local subscription ID {$localSubscription->id} to status {$stripeSubscription->status}");
            } elseif ($user && $plan) { // Create if it doesn't exist (e.g. from checkout.session.completed)
                 Subscription::create([
                    'user_id' => $user->id,
                    'subscription_plan_id' => $plan->id, // This needs to be reliably determined
                    'payment_gateway' => 'stripe',
                    'gateway_subscription_id' => $stripeSubscription->id,
                    'status' => $stripeSubscription->status,
                    'starts_at' => \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_start),
                    'ends_at' => \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end),
                    'trial_ends_at' => $stripeSubscription->trial_end ? \Carbon\Carbon::createFromTimestamp($stripeSubscription->trial_end) : null,
                ]);
                Log::info("Stripe Webhook: Created new local subscription for user {$user->id}, Stripe sub ID {$stripeSubscription->id}");
            } else {
                 Log::warning("Stripe Webhook: Could not create or update local subscription for Stripe sub ID {$stripeSubscription->id}. User or Plan missing.");
            }

        } catch (\Exception $e) {
            Log::error("Stripe Webhook handleSubscriptionUpdate Error for sub ID {$stripeSubscriptionId}: " . $e->getMessage());
        }
    }

    protected function handleSubscriptionCancellation($stripeSubscriptionId)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscriptionId)->first();
        if ($localSubscription) {
            $localSubscription->status = 'cancelled'; // Or 'ended' depending on Stripe's final status
            $localSubscription->cancelled_at = now();
            // ends_at might already be set if it was cancel_at_period_end
            if (!$localSubscription->ends_at || $localSubscription->ends_at->isFuture()) {
                 // If Stripe cancels it immediately, set ends_at to now.
                 // If it was cancel_at_period_end, ends_at should reflect that.
                 // The customer.subscription.updated event with status 'canceled' often provides the final end date.
            }
            $localSubscription->save();
            Log::info("Stripe Webhook: Cancelled local subscription ID {$localSubscription->id}");
        } else {
            Log::warning("Stripe Webhook: Received cancellation for unknown Stripe subscription ID: {$stripeSubscriptionId}");
        }
    }

    protected function handleSubscriptionPaymentFailure($stripeSubscriptionId)
    {
        $localSubscription = Subscription::where('gateway_subscription_id', $stripeSubscriptionId)->first();
        if ($localSubscription) {
            // Stripe might set status to 'past_due' or 'unpaid'
            // You might want to fetch the latest status from Stripe API if not provided directly in event
            if (!setting('stripe_secret_key')) {
                 Log::error('Stripe secret key not set for fetching subscription status on payment failure.');
                 return;
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            try {
                $stripeSub = StripeSubscriptionObject::retrieve($stripeSubscriptionId);
                $localSubscription->status = $stripeSub->status; // e.g., 'past_due'
                $localSubscription->save();
                Log::info("Stripe Webhook: Updated local subscription ID {$localSubscription->id} to status {$stripeSub->status} due to payment failure.");
                // TODO: Notify user about payment failure
            } catch (\Exception $e) {
                Log::error("Stripe Webhook: Error fetching Stripe subscription {$stripeSubscriptionId} on payment failure: " . $e->getMessage());
            }
        } else {
            Log::warning("Stripe Webhook: Received payment failure for unknown Stripe subscription ID: {$stripeSubscriptionId}");
        }
    }
}
