<?php

namespace Modules\StripeGateway\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Modules\StripeGateway\Services\StripeService;
use Stripe\Checkout\Session as StripeCheckoutSession;

class StripeSubscriptionController extends Controller
{
    protected StripeService $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    public function checkout(Request $request, SubscriptionPlan $plan)
    {
        if (setting('stripe_enabled', '0') != '1') {
            return redirect()->route('subscription.plans')->with('error', 'Stripe payments are currently disabled.');
        }

        /** @var User $user */
        $user = Auth::user();

        // Check if user already has an active subscription to this plan or any plan
        // (You might want more sophisticated logic here, e.g., allow upgrades/downgrades)
        $activeSubscription = $user->subscriptions()
                                ->whereIn('status', ['active', 'trialing'])
                                ->first();

        if ($activeSubscription) {
            // If they are trying to subscribe to the same plan they already have and it's active/trialing
            if ($activeSubscription->subscription_plan_id == $plan->id) {
                 return redirect()->route('subscription.plans')->with('info', 'You are already subscribed to this plan.');
            }
            // For now, prevent new subscriptions if one is active.
            // Later, you can implement upgrade/downgrade logic.
            return redirect()->route('subscription.plans')->with('error', 'You already have an active subscription. Please manage it from your profile.');
        }


        try {
            $checkoutSession = $this->stripeService->createCheckoutSession($user, $plan);
            return redirect($checkoutSession->url);
        } catch (\Exception $e) {
            Log::error("Stripe Checkout Error for user {$user->id}, plan {$plan->id}: " . $e->getMessage());
            return redirect()->route('subscription.plans')->with('error', 'Could not initiate Stripe checkout. ' . $e->getMessage());
        }
    }

    public function handleSuccess(Request $request)
    {
        $sessionId = $request->query('session_id');
        if (!$sessionId) {
            return redirect()->route('subscription.plans')->with('error', 'Invalid Stripe session.');
        }

        try {
            if (!setting('stripe_secret_key')) {
                 throw new \Exception('Stripe secret key is not configured.');
            }
            \Stripe\Stripe::setApiKey(setting('stripe_secret_key'));
            $session = StripeCheckoutSession::retrieve($sessionId, ['expand' => ['subscription', 'line_items.data.price.product']]);

            $user = User::find($session->metadata->user_id);
            $plan = SubscriptionPlan::find($session->metadata->plan_id);

            if (!$user || !$plan) {
                Log::error("Stripe success: User or Plan not found from session metadata.", ['session_id' => $sessionId, 'metadata' => $session->metadata]);
                return redirect()->route('subscription.plans')->with('error', 'Subscription details could not be verified.');
            }

            // Check if a subscription record already exists for this Stripe subscription ID
            // This can happen if the webhook processed it first
            $existingSubscription = Subscription::where('gateway_subscription_id', $session->subscription->id)->first();

            if ($existingSubscription) {
                Log::info("Stripe success: Subscription already processed by webhook for Stripe sub ID: " . $session->subscription->id);
                 // Potentially update status if it was pending, though webhook should handle this.
                if ($existingSubscription->status === 'pending') {
                    $existingSubscription->status = $session->subscription->status; // e.g., 'active' or 'trialing'
                    $existingSubscription->starts_at = $session->subscription->current_period_start ? \Carbon\Carbon::createFromTimestamp($session->subscription->current_period_start) : now();
                    $existingSubscription->ends_at = $session->subscription->current_period_end ? \Carbon\Carbon::createFromTimestamp($session->subscription->current_period_end) : null;
                    if ($session->subscription->trial_end) {
                        $existingSubscription->trial_ends_at = \Carbon\Carbon::createFromTimestamp($session->subscription->trial_end);
                    }
                    $existingSubscription->save();
                }
            } else {
                // Create new local subscription record
                Subscription::create([
                    'user_id' => $user->id,
                    'subscription_plan_id' => $plan->id,
                    'payment_gateway' => 'stripe',
                    'gateway_subscription_id' => $session->subscription->id,
                    'status' => $session->subscription->status, // e.g., 'active' or 'trialing'
                    'starts_at' => $session->subscription->current_period_start ? \Carbon\Carbon::createFromTimestamp($session->subscription->current_period_start) : now(),
                    'ends_at' => $session->subscription->current_period_end ? \Carbon\Carbon::createFromTimestamp($session->subscription->current_period_end) : null,
                    'trial_ends_at' => $session->subscription->trial_end ? \Carbon\Carbon::createFromTimestamp($session->subscription->trial_end) : null,
                ]);
            }
            
            // Update user's Stripe customer ID if it's not set (though StripeService should do this)
            if ($user && $session->customer && !$user->stripe_customer_id) {
                $user->stripe_customer_id = $session->customer;
                $user->save();
            }


            return redirect()->route('dashboard')->with('success', 'Subscription successful! Welcome to ' . $plan->name . '.');

        } catch (\Exception $e) {
            Log::error("Stripe Success Error: " . $e->getMessage(), ['session_id' => $sessionId]);
            return redirect()->route('subscription.plans')->with('error', 'There was an issue confirming your subscription: ' . $e->getMessage());
        }
    }

    public function handleCancel(Request $request)
    {
        return redirect()->route('subscription.plans')->with('info', 'Your subscription process was cancelled.');
    }
    
    public function redirectToCustomerPortal(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        if (!$user->stripe_customer_id) {
            return redirect()->back()->with('error', 'Stripe customer profile not found.');
        }

        try {
            $portalSession = $this->stripeService->createBillingPortalSession($user);
            return redirect($portalSession->url);
        } catch (\Exception $e) {
            Log::error("Stripe Billing Portal Error for user {$user->id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'Could not access the billing portal: ' . $e->getMessage());
        }
    }

    public function cancelActiveSubscription(Request $request)
    {
        /** @var User $user */
        $user = Auth::user();
        $activeSubscription = $user->subscriptions()
            ->where('payment_gateway', 'stripe')
            ->whereIn('status', ['active', 'trialing'])
            ->orderBy('created_at', 'desc')
            ->first();

        if (!$activeSubscription) {
            return redirect()->back()->with('info', 'No active Stripe subscription found to cancel.');
        }

        try {
            $cancelledAtGateway = $this->stripeService->cancelSubscriptionAtGateway($activeSubscription);
            if ($cancelledAtGateway) {
                // Webhook should ideally handle the final status update,
                // but we can mark it as 'cancelled' locally for immediate feedback.
                $activeSubscription->status = 'cancelled'; // Or 'pending_cancellation'
                $activeSubscription->cancelled_at = now();
                // ends_at will be updated by webhook when Stripe confirms cancellation at period end
                $activeSubscription->save();
                return redirect()->back()->with('success', 'Your subscription has been scheduled for cancellation at the end of the current billing period.');
            } else {
                return redirect()->back()->with('error', 'Failed to cancel subscription at Stripe. Please try again or contact support.');
            }
        } catch (\Exception $e) {
            Log::error("Error cancelling active Stripe subscription for user {$user->id}, sub ID {$activeSubscription->gateway_subscription_id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while trying to cancel your subscription: ' . $e->getMessage());
        }
    }
}
