<?php

namespace Modules\StripeGateway\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting; // Assuming your Setting model is in App\Models
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class StripeConfigController extends Controller
{
    protected $settingKeys = [
        'stripe_enabled',
        'stripe_publishable_key',
        'stripe_secret_key',
        'stripe_webhook_secret',
        // Add 'stripe_mode' (live/test) if you want to manage this via settings
    ];

    public function edit()
    {
        if (!function_exists('setting')) {
            return redirect()->route('admin.dashboard')->with('error', 'Settings helper function not found.');
        }

        $settings = [];
        foreach ($this->settingKeys as $key) {
            $settings[$key] = setting($key);
        }
        return view('stripegateway::admin.config', compact('settings'));
    }

    public function update(Request $request)
    {
        if (!function_exists('setting')) {
            return redirect()->route('admin.dashboard')->with('error', 'Settings helper function not found.');
        }

        $validated = $request->validate([
            'stripe_enabled' => 'nullable|boolean',
            'stripe_publishable_key' => 'nullable|string|max:255',
            'stripe_secret_key' => 'nullable|string|max:255',
            'stripe_webhook_secret' => 'nullable|string|max:255',
        ]);

        try {
            foreach ($this->settingKeys as $key) {
                $value = $request->has($key) ? ($key === 'stripe_enabled' ? (bool)$request->input($key) : $request->input($key)) : ($key === 'stripe_enabled' ? '0' : null);
                
                // For boolean 'stripe_enabled', ensure '0' or '1' is stored if it's a checkbox
                if ($key === 'stripe_enabled') {
                    $valueToStore = $request->has('stripe_enabled') ? '1' : '0';
                } else {
                    $valueToStore = $request->input($key);
                }

                Setting::updateOrCreate(
                    ['key' => $key],
                    ['value' => $valueToStore]
                );
            }

            // Clear cache for settings to take effect immediately
            Artisan::call('cache:clear');
            Artisan::call('config:clear');


            return redirect()->back()->with('success', 'Stripe settings updated successfully.');

        } catch (\Exception $e) {
            Log::error('Error updating Stripe settings: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update Stripe settings. Please check the logs.');
        }
    }
}
